package com.xunyi.micro.propagation.instrument.reactor;

import brave.Tracing;
import com.xunyi.micro.propagation.context.CurrentContext;
import lombok.extern.slf4j.Slf4j;
import org.reactivestreams.Publisher;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.context.ConfigurableApplicationContext;
import reactor.core.CoreSubscriber;
import reactor.core.Fuseable;
import reactor.core.Scannable;
import reactor.core.publisher.Operators;
import reactor.util.context.Context;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BooleanSupplier;
import java.util.function.Function;

@Slf4j
public class Reactors {


    public static <T> Function<? super Publisher<T>, ? extends Publisher<T>> scopePassingSpanOperator(
            BeanFactory beanFactory) {

        // Adapt if lazy bean factory
        BooleanSupplier isActive = beanFactory instanceof ConfigurableApplicationContext
                ? ((ConfigurableApplicationContext) beanFactory)::isActive : () -> true;

        return Operators.liftPublisher((p, sub) -> {
            // if Flux/Mono #just, #empty, #error
            if (p instanceof Fuseable.ScalarCallable) {
                return sub;
            }
            Scannable scannable = Scannable.from(p);
            // rest of the logic unchanged...
            if (isActive.getAsBoolean()) {
                if (log.isTraceEnabled()) {
                    log.trace("Spring Context [{}}] already refreshed. Creating a scope "
                            + "passing propagation subscriber with Reactor Context " + "[{}] and name [{}]",
                            beanFactory, sub.currentContext(), scannable.name());
                }

                return scopePassingSpanSubscription(beanFactory, sub);
            }
            if (log.isTraceEnabled()) {
                log.trace("Spring Context [" + beanFactory
                        + "] is not yet refreshed, falling back to lazy span subscriber. Reactor Context is ["
                        + sub.currentContext() + "] and name is [" + scannable.name()
                        + "]");
            }
            return new LazyPassingSubscriber<T>(
                    lazyScopePassingSubscription(beanFactory, scannable, sub));
        });
    }

    static <T> PassingSubscriptionProvider<T> lazyScopePassingSubscription(
            BeanFactory beanFactory, Scannable scannable, CoreSubscriber<? super T> sub) {
        return new PassingSubscriptionProvider<>(beanFactory, sub, sub.currentContext());
    }
    private static Map<BeanFactory, CurrentContext> CACHE = new ConcurrentHashMap<>();
    static <T> CoreSubscriber<? super T> scopePassingSpanSubscription(
            BeanFactory beanFactory, CoreSubscriber<? super T> sub) {
        CurrentContext currentContext = CACHE.computeIfAbsent(beanFactory,
                beanFactory1 -> beanFactory1.getBean(CurrentContext.class));
        Context context = sub.currentContext();

        com.xunyi.micro.propagation.context.Context root = context.hasKey(com.xunyi.micro.propagation.context.Context.class) ? context.get(com.xunyi.micro.propagation.context.Context.class)
                : currentContext.get();
        if (root != null) {
            return new ScopePassingSubscriber<>(sub, context, currentContext, root);
        }
        else {
            return sub; // no need to trace
        }
    }
}
