package com.xunyi.micro.jackson;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.mongodb.MongoClient;
import com.xunyi.micro.data.page.PageSerializer;
import com.xunyi.micro.data.repository.mongo.MongoModule;
import com.xunyi.micro.message.jackson.MessageModule;
import com.xunyi.money.jackson.MoneyModule;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.boot.autoconfigure.jackson.JacksonAutoConfiguration;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.domain.Page;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.util.ClassUtils;


/**
 * 可以实现自定义序列化Bean, JsonComponentModule会加载
 *
 * @see org.springframework.boot.autoconfigure.jackson.JacksonAutoConfiguration
 */
@Configuration
@ConditionalOnClass(ObjectMapper.class)
@AutoConfigureBefore(JacksonAutoConfiguration.class)
public class JacksonCustomAutoConfiguration {

    @Configuration
    @ConditionalOnClass(Jackson2ObjectMapperBuilder.class)
    static class JacksonCustomizerConfiguration {
        @Bean
        public Jackson2ObjectMapperBuilderCustomizer jackson2ObjectMapperBuilderCustomizer() {
            return jacksonObjectMapperBuilder -> {
                jacksonObjectMapperBuilder.serializationInclusion(JsonInclude.Include.NON_NULL);
                jacksonObjectMapperBuilder.featuresToEnable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
                jacksonObjectMapperBuilder.featuresToDisable(SerializationFeature.WRITE_DATE_TIMESTAMPS_AS_NANOSECONDS);
                jacksonObjectMapperBuilder.featuresToDisable(DeserializationFeature.READ_DATE_TIMESTAMPS_AS_NANOSECONDS);
            };
        }

        @ConditionalOnClass(Page.class)
        @Bean
        public Jackson2ObjectMapperBuilderCustomizer pageJackson2ObjectMapperBuilderCustomizer() {
            return jacksonObjectMapperBuilder -> {
                jacksonObjectMapperBuilder.serializers(new PageSerializer());
            };
        }

        @Bean
        @ConditionalOnBean(MessageSource.class)
        public MessageModule messageModule(MessageSource source) {
            return new MessageModule(source);
        }

        @Bean
        @ConditionalOnBean(MongoClient.class)
        public MongoModule mongoModule() {
            return new MongoModule();
        }

        @Bean
        public MoneyModule moneyModule() {
            return new MoneyModule();
        }
    }

}
