package com.xunyi.micro.web.bind;


import com.xunyi.micro.message.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.Iterator;
import java.util.Set;

@Slf4j
public abstract class AbstractErrorControllerAdvice {

    public abstract IErrorOwner errorOfCode(String code) throws Throwable;


    protected IErrorOwner errorOfMessage(String message, IErrorOwner defaultError) {
        try {
            return errorOfCode(message);
        } catch (Throwable e) {
            log.error("not match error message: ", e);
            return defaultError;
        }
    }
    protected Return returnOfMessage(String message, IErrorOwner defaultError) {
        return errorOfMessage(message, defaultError).toReturn();
    }
    protected Return returnOfMessage(String message) {
       return returnOfMessage(message, StandardErrors.UNKNOWN_ERROR);
    }


    @ExceptionHandler(ConstraintViolationException.class)
    public @ResponseBody
    Return constraintViolationException(ConstraintViolationException cve) {
        Set<ConstraintViolation<?>> cvs = cve.getConstraintViolations();
        Iterator<ConstraintViolation<?>> cvIterator = cvs.iterator();
        ConstraintViolation<?> cs = cvIterator.next();
        String message = cs.getMessage();
        return errorOfMessage(message, StandardErrors.PARAMETER_BAD).toReturn(cs.getPropertyPath().toString());
    }


    protected Return returnOfBindResult(BindingResult bindingResult) {
        FieldError fieldError = bindingResult.getFieldError();
        log.error("returnOfBindResult: Field Error{}", fieldError);
        assert fieldError != null;
        String message = fieldError.getDefaultMessage();
        return errorOfMessage(message, StandardErrors.PARAMETER_BAD).toReturn(fieldError.getField());
    }

    @ExceptionHandler(MethodArgumentNotValidException.class)
    public @ResponseBody Return methodArgumentNotValidException(MethodArgumentNotValidException e) {
        log.error("handle MethodArgumentNotValidException", e);
        return returnOfBindResult(e.getBindingResult());
    }

    @ExceptionHandler(MethodArgumentTypeMismatchException.class)
    public @ResponseBody Return handleMethodArgumentTypeMismatchException(
            HttpServletRequest request,
            MethodArgumentTypeMismatchException e
    ) {
        log.error("handle MissingRequestParameter url:{} name: {}", request.getRequestURI(), e.getName(), e);
        return StandardErrors.PARAMETER_BAD.toReturn(e.getName());
    }

    @ExceptionHandler(MissingServletRequestParameterException.class)
    public @ResponseBody Return handleMissingRequestParameter(
            HttpServletRequest request,
            MissingServletRequestParameterException e
    ) {
        log.error("handle MissingRequestParameter url:{} name: {}", request.getRequestURI(), e.getParameterName(), e);
        return StandardErrors.PARAMETER_BAD.toReturn(e.getParameterName());
    }

    @ExceptionHandler(BindException.class)
    public @ResponseBody Return bindException(BindException e) {
        log.error("handle BindException message:{}", e.getMessage(), e);
        return returnOfBindResult(e.getBindingResult());
    }

    @ExceptionHandler(ErrorException.class)
    public @ResponseBody Return handleErrorException(ErrorException e) {
        log.error("handle ErrorException error:{}", e.getError());
        return e.toReturn();
    }

    @ExceptionHandler(Exception.class)
    public @ResponseBody Return handleException(Exception e) {
        log.error("handleException", e);
        return StandardErrors.UNKNOWN_ERROR.toReturn();
    }

}
