package com.xunyi.micro.shunt.propagation;


import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.xunyi.micro.propagation.Propagation;
import com.xunyi.micro.propagation.context.Extractor;
import com.xunyi.micro.propagation.context.Injector;

import java.util.Collections;
import java.util.List;

/**
 * 分流传播
 * @param <K>
 */
public class ShuntPropagation<K> implements Propagation<K> {


    private final K sequencesKey;
    private final List<K> fields;

    public ShuntPropagation(KeyFactory<K> keyFactory) {
        this.sequencesKey = keyFactory.create("X-Sequences");
        this.fields = Collections.unmodifiableList(
                Collections.singletonList(this.sequencesKey)
        );
    }

    @Override
    public List<K> keys() {
        return fields;
    }

    @Override
    public <C> Injector<C> injector(Setter<C, K> setter) {
        return new InnerInjector<C, K>(this, setter);
    }

    @Override
    public <C> Extractor<C> extractor(Getter<C, K> getter) {
        return new InnerExtractor<C, K>(this, getter);
    }


    static final class InnerInjector<C, K> implements Injector<C> {
        private ShuntPropagation<K> propagation;
        private Setter<C, K> setter;

        private InnerInjector(ShuntPropagation<K> propagation, Setter<C, K> setter) {
            this.propagation = propagation;
            this.setter = setter;
        }
        @Override
        public <T> void inject(T context, C carrier) {
            ShuntContext shuntContext = (ShuntContext) context;
            this.setter.put(carrier, this.propagation.sequencesKey, Joiner.on(",").join(shuntContext.sequences()));
        }
    }

    static final class InnerExtractor<C, K> implements Extractor<C> {
        private ShuntPropagation<K> propagation;
        private Getter<C, K> getter;

        private InnerExtractor(ShuntPropagation<K> propagation, Getter<C, K> getter) {
            this.propagation = propagation;
            this.getter = getter;
        }

        @Override
        public <T> T extract(C carrier) {
            String sequencesString = getter.get(carrier, propagation.sequencesKey);

            List<String> sequences = Collections.emptyList();
            if (!Strings.isNullOrEmpty(sequencesString)) {
                sequences = Splitter.on(",").splitToList(sequencesString);
            }
            return (T) new ShuntContext(sequences);
        }
    }
}
