package com.xunyi.micro.grpc.discovery;

import com.google.common.collect.Lists;
import io.grpc.Attributes;
import io.grpc.EquivalentAddressGroup;
import io.grpc.NameResolver;
import io.grpc.Status;
import io.grpc.internal.SharedResourceHolder;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.Equator;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.discovery.DiscoveryClient;
import org.springframework.cloud.client.discovery.event.HeartbeatEvent;
import org.springframework.cloud.client.discovery.event.HeartbeatMonitor;
import org.springframework.context.event.EventListener;

import java.net.InetSocketAddress;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Function;
import java.util.stream.Collectors;


public class DiscoveryNameResolver extends NameResolver {

    private final String name;
    private final DiscoveryClient client;
    private final SharedResourceHolder.Resource<ScheduledExecutorService> timerServiceResource;
    private ScheduledExecutorService timerServiceHolder;

    private Listener listener;

    private volatile boolean resolving;
    private volatile boolean shutdown;

    public DiscoveryNameResolver(String name, DiscoveryClient client, Attributes attributes, SharedResourceHolder.Resource<ScheduledExecutorService> timerServiceResource) {
        this.name = name;
        this.client = client;
        this.timerServiceResource = timerServiceResource;
    }

    //可考虑统一管理name resolver
    private HeartbeatMonitor monitor = new HeartbeatMonitor();
    @EventListener(HeartbeatEvent.class)
    public void heartbeat(HeartbeatEvent event) {
        if (this.monitor.update(event.getValue())) {
           this.refresh();
        }
    }

    @Override
    public String getServiceAuthority() {
        return this.name;
    }

    @Override
    public void start(Listener listener) {
        //listner
        this.listener = listener;
        timerServiceHolder = SharedResourceHolder.get(timerServiceResource);
        //是否定时查询
//        timerService.scheduleAtFixedRate()
        resolve();
    }

    @Override
    public final synchronized void refresh() {
        if (listener != null) {
            resolve();
        }
    }

    @Override
    public void shutdown() {
        if (shutdown) {
            return;
        }
        shutdown = true;
        if (timerServiceHolder != null) {
            timerServiceHolder = SharedResourceHolder.release(timerServiceResource, timerServiceHolder);
        }
    }

    private void resolve() {
        if (resolving || shutdown) {
            return;
        }

        //是否需异步执行
        resolveRunnable.run();
    }

    public final Runnable resolveRunnable = new Runnable() {

        private List<ServiceInstance> serviceInstances = Lists.newArrayList();

        @Override
        public void run() {
            Listener savedListener;
            synchronized (DiscoveryNameResolver.this) {
                if (shutdown) {
                    return;
                }
                if (resolving) {
                    return;
                }
                resolving = true;
            }
            savedListener = listener;
            try {
                List<ServiceInstance> newServiceInstanceList;
                try {
                    newServiceInstanceList = client.getInstances(name);
                } catch (Exception e) {
                    savedListener.onError(Status.INVALID_ARGUMENT.withCause(e));
                    return;
                }

                if (CollectionUtils.isNotEmpty(newServiceInstanceList)) {
                    if (isServiceInstanceListUpdate(newServiceInstanceList)) {
                        return;
                    }
                    serviceInstances = newServiceInstanceList;

                    List<EquivalentAddressGroup> equivalentAddressGroups = serviceInstances.stream().map(new Function<ServiceInstance, EquivalentAddressGroup>() {
                        @Override
                        public EquivalentAddressGroup apply(ServiceInstance serviceInstance) {
                            return new EquivalentAddressGroup(
                                    new InetSocketAddress(serviceInstance.getHost(), serviceInstance.getPort())
                            );
                        }
                    }).collect(Collectors.toList());
                    listener.onAddresses(equivalentAddressGroups, Attributes.EMPTY);
                } else {
                    savedListener.onError(Status.UNAVAILABLE.withCause(new RuntimeException("UNAVAILABLE: NameResolver returned an empty list")));
                }
            } finally {
                synchronized (DiscoveryNameResolver.this) {
                    resolving = false;
                }
            }
        }


        private boolean isServiceInstanceListUpdate(List<ServiceInstance> newServiceInstances) {
            return CollectionUtils.isEqualCollection(newServiceInstances, serviceInstances, new Equator<ServiceInstance>() {
                @Override
                public boolean equate(ServiceInstance serviceInstance, ServiceInstance t1) {
                    return Objects.equals(serviceInstance.getHost(), t1.getHost()) &&
                            Objects.equals(serviceInstance.getPort(), t1.getPort());
                }

                @Override
                public int hash(ServiceInstance serviceInstance) {
                    return serviceInstance.getPort();
                }
            });
        }
    };


}
