package com.xunyi.beast.web.servlet.error;

import com.xunyi.beast.data.message.*;
import com.xunyi.beast.web.bind.ErrorOwnerProvider;
import com.xunyi.beast.web.support.ServerExchangeUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.catalina.connector.Response;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.UnsatisfiedServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.util.pattern.PathPattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.Path;
import java.io.IOException;
import java.util.Iterator;
import java.util.Objects;
import java.util.Set;

@Slf4j
@RestControllerAdvice
public class ConstraintExceptionAdvice extends AbstractExceptionAdvice {

    private ErrorMessageSource errorMessageSource;

    private ErrorOwnerProvider errorOwnerProvider;

    public ConstraintExceptionAdvice(ErrorOwnerProvider errorOwnerProvider, ErrorMessageSource errorMessageSource) {
        super(errorMessageSource);
        this.errorOwnerProvider = errorOwnerProvider;
        this.errorMessageSource = errorMessageSource;
    }

    protected IErrorOwner errorOfMessage(String message, IErrorOwner defaultError) {
        IErrorOwner owner = null;
        try {
            owner = errorOwnerProvider.ofCode(message);
        } catch (Throwable ignore) {
        }
        if (owner == null) {
            owner = defaultError;
        }
        return owner;
    }



    @ExceptionHandler({
            MissingServletRequestParameterException.class,
            MethodArgumentTypeMismatchException.class
    })
    public ModelAndView missingRequestParameter(
            HttpServletRequest request, HttpServletResponse response,
            Exception e
    ) throws IOException {
        String code = null;
        Object[] args = null;
        if (e instanceof MissingServletRequestParameterException) {
            MissingServletRequestParameterException missingServletRequestParameterException = (MissingServletRequestParameterException) e;
            String missingParameterName = missingServletRequestParameterException.getParameterName();
            String missingParameterType = missingServletRequestParameterException.getParameterType();
            code = StandardErrors.PARAMETER_MISSING.getErrorCode();
            args = new Object[]{missingParameterType, missingParameterName};

        } else if (e instanceof MethodArgumentTypeMismatchException) {
            MethodArgumentTypeMismatchException methodArgumentTypeMismatchException = (MethodArgumentTypeMismatchException) e;
            code = StandardErrors.PARAMETER_TYPE_MISMATCH.getErrorCode();
            String requiredType = Objects.requireNonNull(methodArgumentTypeMismatchException.getRequiredType()).getSimpleName();
            args = new Object[]{methodArgumentTypeMismatchException.getName(), requiredType};
        }
        String message = errorMessageSource.getMessage(code, args);
        request.setAttribute(ServerExchangeUtils.ERROR_ATTR, code);
        request.setAttribute(ServerExchangeUtils.MESSAGE_ATTR, message);
        response.sendError(Response.SC_OK);
        return new ModelAndView();
    }

    @ExceptionHandler({
            ConstraintViolationException.class
    })
    public ModelAndView constraintViolationException(HttpServletRequest request, HttpServletResponse response, ConstraintViolationException cve) throws IOException {
        Set<ConstraintViolation<?>> cvs = cve.getConstraintViolations();
        Iterator<ConstraintViolation<?>> cvIterator = cvs.iterator();
        ConstraintViolation<?> cs = cvIterator.next();
        Path propertyPath = cs.getPropertyPath();
        String message = cs.getMessage();
        log.warn("property-path: {} message: {}", propertyPath, message);
        return returnByMessage(message, request, response);
    }






    @ExceptionHandler(UnsatisfiedServletRequestParameterException.class)
    public ModelAndView unsatisfiedServletRequestParameterException(
            HttpServletRequest request, HttpServletResponse response,
            UnsatisfiedServletRequestParameterException e
    ) throws IOException {
        log.warn("{} {}", getRequestInfo(request), e.getMessage(), e);
        return returnErrorMessage(StandardErrors.BAD_PARAMETER.toError(), request, response);
    }




    @ExceptionHandler(MethodArgumentNotValidException.class)
    public ModelAndView methodArgumentNotValidException(
            HttpServletRequest request, HttpServletResponse response,
            MethodArgumentNotValidException e
    ) throws IOException {
        log.warn("{} method argument not valid:", getRequestInfo(request), e);
        return returnByBindingResult(e.getBindingResult(), request, response);
    }


    @ExceptionHandler(BindException.class)
    public @ResponseBody ModelAndView bindException(
            HttpServletRequest request, HttpServletResponse response,
            BindException e
    ) throws IOException {
        log.warn("{} bind exception", getRequestInfo(request), e);
        return returnByBindingResult(e.getBindingResult(), request, response);
    }




    protected ModelAndView returnByBindingResult(BindingResult bindingResult, HttpServletRequest request, HttpServletResponse response) throws IOException {
        FieldError fieldError = bindingResult.getFieldError();

        assert fieldError != null;
        String message = fieldError.getDefaultMessage();
        String field = fieldError.getField();
        log.warn("returnByBindingResult: Field field:{} Error{}", field, fieldError);
        if (message == null) {
            return returnErrorMessage(StandardErrors.BAD_PARAMETER.toError(), request, response);
        }
        return returnByMessage(message, request, response);
    }



    public ModelAndView returnByMessage(String message, HttpServletRequest request, HttpServletResponse response) throws IOException {
        IErrorOwner owner = errorOfMessage(message, StandardErrors.BAD_PARAMETER);
        String code;
        if (owner == StandardErrors.BAD_PARAMETER) {
            code = owner.getErrorCode();
        } else {
            code = owner.getErrorCode();
            message = errorMessageSource.getMessage(owner.toError());
        }
        return returnErrorMessage(code, message, request, response);
    }


}
