package com.xunyi.beast.web.servlet.error;

import com.google.common.base.Strings;
import com.xunyi.beast.data.message.*;
import com.xunyi.beast.web.support.ServerExchangeUtils;
import feign.RetryableException;
import lombok.extern.slf4j.Slf4j;
import org.apache.catalina.connector.Response;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.core.Ordered;
import org.springframework.data.domain.Sort;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.lang.Nullable;
import org.springframework.validation.BindException;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingPathVariableException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.context.request.async.AsyncRequestTimeoutException;
import org.springframework.web.multipart.support.MissingServletRequestPartException;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.NoHandlerFoundException;
import org.springframework.web.servlet.handler.AbstractHandlerExceptionResolver;
import org.springframework.web.servlet.mvc.support.DefaultHandlerExceptionResolver;
import org.springframework.web.util.pattern.PathPattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;


@Slf4j
public class BeastHandlerExceptionResolver extends AbstractHandlerExceptionResolver  {

    private ErrorMessageSource errorMessageSource;

    public BeastHandlerExceptionResolver(ErrorMessageSource errorMessageSource) {
        this.errorMessageSource = errorMessageSource;
        setWarnLogCategory(getClass().getName());
    }

    @Override
    protected ModelAndView doResolveException(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
        try {
            if (ex instanceof ErrorException) {
                return handleError((ErrorException) ex, request, response, handler);
            }
            return handleException(ex, request, response, handler);
        } catch (Exception handlerEx) {
            logger.warn("Failure while trying to resolve exception [" + ex.getClass().getName() + "]", handlerEx);
        }
        return null;
    }


    public ModelAndView handleError(ErrorException ex,
        HttpServletRequest request, HttpServletResponse response, @Nullable Object handler) throws IOException {
        if (log.isDebugEnabled()) {
            log.debug("handleError:" + ex.getMessage(), ex);
        }
        IError error = ex.getError();
        String message = null;
        if (error instanceof ErrorInfo) {
            message = ((ErrorInfo) error).getMessage();
        }
        if (Strings.isNullOrEmpty(message)) {
            message = errorMessageSource.getMessage(error);
        }
        request.setAttribute(ServerExchangeUtils.ERROR_ATTR, error.getCode());
        request.setAttribute(ServerExchangeUtils.MESSAGE_ATTR, message);
        response.sendError(Response.SC_OK);
        return new ModelAndView();
    }

    private static final String DATA_REST_PATH_PATTERN_ATTRIBUTE = "org.springframework.data.rest.webmvc.RepositoryRestHandlerMapping.EFFECTIVE_REPOSITORY_RESOURCE_LOOKUP_PATH";
    private @Nullable static String getMatchingPattern(HttpServletRequest request) {
        PathPattern dataRestPathPattern = (PathPattern) request.getAttribute(DATA_REST_PATH_PATTERN_ATTRIBUTE);
        if (dataRestPathPattern != null) {
            return dataRestPathPattern.getPatternString();
        }
        return (String) request.getAttribute(HandlerMapping.BEST_MATCHING_PATTERN_ATTRIBUTE);
    }

    public ModelAndView handleException(Exception ex, HttpServletRequest request, HttpServletResponse response, @Nullable Object handler) throws IOException {
        String method = request.getMethod();
        String pattern = getMatchingPattern(request);
        String uri = "UNKNOWN";
        if (pattern != null) {
            //if (ignoreTrailingSlash) 忽略尾部斜杠
            if (pattern.isEmpty()) {
                uri = "ROOT";
            } else {
                uri = pattern;
            }
        }
        log.warn("request [{} {}] resolved:[{}]", method, uri, ex.getMessage(), ex);
        IError error = StandardErrors.SERVICE_INTERNAL_ERROR.toError();
        String message = errorMessageSource.getMessage(error);
        request.setAttribute(ServerExchangeUtils.ERROR_ATTR, error.getCode());
        request.setAttribute(ServerExchangeUtils.MESSAGE_ATTR, message);
        response.sendError(Response.SC_OK);
        return new ModelAndView();
    }

}
