package com.xunyi.beast.web.servlet.error;

import com.xunyi.beast.web.support.ServerExchangeUtils;
import org.springframework.boot.web.error.ErrorAttributeOptions;
import org.springframework.boot.web.servlet.error.ErrorAttributes;
import org.springframework.core.Ordered;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.RequestDispatcher;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.time.OffsetDateTime;
import java.util.LinkedHashMap;
import java.util.Map;

public class BeastErrorAttributes implements ErrorAttributes, HandlerExceptionResolver, Ordered {

    public static final String ERROR_ATTRIBUTE = BeastErrorAttributes.class.getName() + ".ERROR";

    @Override
    public Map<String, Object> getErrorAttributes(WebRequest webRequest, ErrorAttributeOptions options) {
        Map<String, Object> errorAttributes = new LinkedHashMap<>();
        errorAttributes.put("timestamp", OffsetDateTime.now());
        addStatus(errorAttributes, webRequest);
        addErrorDetail(errorAttributes, webRequest);
        addPath(errorAttributes, webRequest);

        //todo: 跟踪TraceID
        return errorAttributes;
    }

    private void addStatus(Map<String, Object> errorAttributes, RequestAttributes requestAttributes) {
        Integer status = getAttribute(requestAttributes, RequestDispatcher.ERROR_STATUS_CODE);
        if (status == null) {
            errorAttributes.put("status", 999);
        }
        errorAttributes.put("status", status);
    }

    private void addErrorDetail(Map<String, Object> errorAttributes, WebRequest webRequest) {
        Throwable throwable = getError(webRequest);
        addExceptionErrorMessage(errorAttributes, webRequest, throwable);
//        addStackTrace(errorAttributes, throwable);
    }

    private void addExceptionErrorMessage(Map<String, Object> errorAttributes, WebRequest webRequest, @Nullable Throwable throwable) {
        String error = getAttribute(webRequest, ServerExchangeUtils.ERROR_ATTR);
        errorAttributes.put("error", error);
        Object message = getAttribute(webRequest, ServerExchangeUtils.MESSAGE_ATTR);
        if (StringUtils.isEmpty(message)) {
            message = getAttribute(webRequest, RequestDispatcher.ERROR_MESSAGE);
        }
        if (StringUtils.isEmpty(message) && throwable != null) {
            message = throwable.getMessage();
        }
        if (StringUtils.isEmpty(message)) {
            message = "No message available";
        }
        errorAttributes.put("message", message);
    }
    private void addStackTrace(Map<String, Object> errorAttributes, Throwable throwable) {
        StringWriter stackTrace = new StringWriter();
        throwable.printStackTrace(new PrintWriter(stackTrace));
        stackTrace.flush();
        errorAttributes.put("trace", stackTrace.toString());

    }

    private void addPath(Map<String, Object> errorAttributes, RequestAttributes requestAttributes) {
        String path = getAttribute(requestAttributes, RequestDispatcher.ERROR_REQUEST_URI);
        if (path != null) {
            errorAttributes.put("path", path);
        }
    }

    @Override
    public Throwable getError(WebRequest webRequest) {
        Throwable exception = getAttribute(webRequest, ERROR_ATTRIBUTE);
        return (exception != null) ? exception : getAttribute(webRequest, RequestDispatcher.ERROR_EXCEPTION);
    }

    @Override
    public ModelAndView resolveException(@NonNull HttpServletRequest request, HttpServletResponse response, Object handler,
                                         @NonNull Exception ex) {
        storeErrorAttributes(request, ex);
        return null;
    }

    private void storeErrorAttributes(HttpServletRequest request, Exception ex) {
        request.setAttribute(ERROR_ATTRIBUTE, ex);
    }

    private <T> T getAttribute(RequestAttributes requestAttributes, String name) {
        return (T) requestAttributes.getAttribute(name, RequestAttributes.SCOPE_REQUEST);
    }

    @Override
    public int getOrder() {
        return Ordered.HIGHEST_PRECEDENCE;
    }

}
