package com.xunyi.beast.web.resolver;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.core.MethodParameter;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.annotation.MethodArgumentConversionNotSupportedException;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;
import org.springframework.web.util.UrlPathHelper;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.lang.annotation.Annotation;


@Slf4j
public abstract class AbstractTokenArgumentResolver<A extends Annotation> implements HandlerMethodArgumentResolver {

    private UrlPathHelper urlPathHelper = new UrlPathHelper();

    private Class<A> annotationClass;

    private String cookieName;

    public AbstractTokenArgumentResolver(String cookieName, Class<A> annotationClass) {
        this.annotationClass = annotationClass;
        this.cookieName = cookieName;
    }

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return parameter.hasParameterAnnotation(this.annotationClass);
    }

    public abstract boolean isRequired(A annotation);

    @Override
    public Object resolveArgument(MethodParameter parameter, @Nullable ModelAndViewContainer mavContainer,
                                  NativeWebRequest webRequest, @Nullable WebDataBinderFactory binderFactory) throws Exception {
        A annotation = parameter.getParameterAnnotation(this.annotationClass);
        Assert.state(annotation != null, "No " + annotationClass.getSimpleName() + " annotation");
        String name = parameter.getParameterName();
        if (name == null) {
            name = "";
        }
        MethodParameter nestedParameter = parameter.nestedIfOptional();

        Object arg = resolveValue(nestedParameter, webRequest);
        if (arg == null) {
            if (isRequired(annotation) && !nestedParameter.isOptional()) {
                handleMissingValue(parameter);
            }
            arg = handleNullValue(arg, nestedParameter.getNestedParameterType());
        }
        if (binderFactory != null) {
            WebDataBinder binder = binderFactory.createBinder(webRequest, null, name);
            try {
                arg = binder.convertIfNecessary(arg, parameter.getParameterType(), parameter);
            }
            catch (ConversionNotSupportedException ex) {
                throw new MethodArgumentConversionNotSupportedException(arg, ex.getRequiredType(),
                        name, parameter, ex.getCause());
            }
            catch (TypeMismatchException ex) {
                throw new MethodArgumentTypeMismatchException(arg, ex.getRequiredType(),
                        name, parameter, ex.getCause());
            }
        }
        return arg;
    }

    protected abstract void handleMissingValue(MethodParameter parameter) throws ServletRequestBindingException;


    @Nullable
    private Object handleNullValue(@Nullable Object value, Class<?> paramType) {
        if (value == null) {
            if (Boolean.TYPE.equals(paramType)) {
                return Boolean.FALSE;
            }
        }
        return value;
    }

    protected abstract Object resolveToken(MethodParameter parameter, String tokenString) throws ServletRequestBindingException;

    private @Nullable Object resolveValue(MethodParameter parameter, NativeWebRequest webRequest) throws ServletRequestBindingException {
        HttpServletRequest servletRequest = webRequest.getNativeRequest(HttpServletRequest.class);
        Assert.state(servletRequest != null, "No HttpServletRequest");
        Cookie cookieValue = WebUtils.getCookie(servletRequest, this.cookieName);
        if (cookieValue == null) {
            return null;
        }
        String tokenString = this.urlPathHelper.decodeRequestString(servletRequest, cookieValue.getValue());

        return resolveToken(parameter, tokenString);
    }

//    @Override
//    public Object resolveArgument(MethodParameter parameter, ModelAndViewContainer mavContainer,
//              NativeWebRequest webRequest, WebDataBinderFactory binderFactory) throws Exception {
////        webRequest.
//        webRequest.getParameter()
//        return null;
//    }
}