package com.xunyi.beast.web.servlet.error;

import com.xunyi.beast.data.message.ErrorMessageSource;
import com.xunyi.beast.data.message.IError;
import com.xunyi.beast.data.message.StandardErrors;
import com.xunyi.beast.web.support.ServerExchangeUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.mvc.support.DefaultHandlerExceptionResolver;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;


@Slf4j
public class InnerHandlerExceptionResolver extends DefaultHandlerExceptionResolver {

    private final ErrorMessageSource errorMessageSource;

    public InnerHandlerExceptionResolver(ErrorMessageSource errorMessageSource) {
        this.errorMessageSource = errorMessageSource;
        setWarnLogCategory(getClass().getName());
    }

    @Override
    protected @NonNull ModelAndView handleHttpMediaTypeNotAcceptable(HttpMediaTypeNotAcceptableException ex, HttpServletRequest request, HttpServletResponse response, Object handler) throws IOException {
        log.warn("request uri:[{}] method: [{}] HttpMediaTypeNotAcceptable", request.getRequestURI(), request.getMethod());
        applyErrorMessage(StandardErrors.HTTP_MEDIA_TYPE_NOT_ACCEPTABLE.toError(), request);
        return super.handleHttpMediaTypeNotAcceptable(ex, request, response, handler);
    }

    @Override
    protected @NonNull ModelAndView handleHttpRequestMethodNotSupported(HttpRequestMethodNotSupportedException ex, HttpServletRequest request, HttpServletResponse response, Object handler) throws IOException {
        log.warn("request uri:[{}] method: [{}] supportedMediaType: [{}] HttpRequestMethodNotSupported", request.getRequestURI(), ex.getMethod(), ex.getSupportedMethods());
        applyErrorMessage(StandardErrors.HTTP_REQUEST_METHOD_NOT_SUPPORTED.toError(), request);
        return super.handleHttpRequestMethodNotSupported(ex, request, response, handler);
    }

    public void applyErrorMessage(IError error, HttpServletRequest request) {
        String message = errorMessageSource.getMessage(error);
        request.setAttribute(ServerExchangeUtils.ERROR_ATTR, error.getCode());
        request.setAttribute(ServerExchangeUtils.MESSAGE_ATTR, message);
    }
}
