package com.xunyi.beast.web.servlet.error;

import com.google.common.collect.Maps;
import com.xunyi.beast.data.message.*;
import com.xunyi.beast.web.support.ServerExchangeUtils;
import feign.FeignException;
import feign.Request;
import feign.RetryableException;
import lombok.extern.slf4j.Slf4j;
import org.apache.catalina.connector.Response;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.condition.SearchStrategy;
import org.springframework.boot.autoconfigure.web.ServerProperties;
import org.springframework.boot.autoconfigure.web.servlet.WebMvcAutoConfiguration;
import org.springframework.boot.autoconfigure.web.servlet.error.ErrorMvcAutoConfiguration;
import org.springframework.boot.autoconfigure.web.servlet.error.ErrorViewResolver;
import org.springframework.boot.web.servlet.error.ErrorAttributes;
import org.springframework.boot.web.servlet.error.ErrorController;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.OrderComparator;
import org.springframework.http.MediaType;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.DispatcherServlet;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.View;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import org.springframework.web.util.HtmlUtils;

import javax.servlet.Servlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.*;
import java.util.stream.Collectors;

@Configuration(proxyBeanMethods = false)
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
@ConditionalOnClass({ Servlet.class, DispatcherServlet.class})
@AutoConfigureBefore({WebMvcAutoConfiguration.class, ErrorMvcAutoConfiguration.class})
public class BeastErrorMvcAutoConfiguration implements WebMvcConfigurer {

    private final ServerProperties serverProperties;

    public BeastErrorMvcAutoConfiguration(ServerProperties serverProperties) {
        this.serverProperties = serverProperties;
    }

    @Bean
    @ConditionalOnMissingBean(value = ErrorAttributes.class, search = SearchStrategy.CURRENT)
    public BeastErrorAttributes beastErrorAttributes() {
        return new BeastErrorAttributes();
    }

    @Bean
    @ConditionalOnMissingBean(value = ErrorController.class, search = SearchStrategy.CURRENT)
    public BeastErrorController basicErrorController(
            ErrorAttributes errorAttributes,
            ObjectProvider<ErrorViewResolver> errorViewResolvers
    ) {
        return new BeastErrorController(errorAttributes, this.serverProperties.getError(), errorViewResolvers.orderedStream().collect(Collectors.toList()));
    }

    @Slf4j
    private static class ErrorView implements View {

        private static final MediaType TEXT_HTML_UTF8 = new MediaType("text", "html", StandardCharsets.UTF_8);

        private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS ZZZ");
        @Override
        public void render(Map<String, ?> model, @NonNull HttpServletRequest request, HttpServletResponse response) throws Exception {
            if (response.isCommitted()) {
                Object path = model.get("path");
                Object message = model.get("message");
                log.error("Cannot render error page for request [{}] and exception [{}] as the response has already been committed", path, message);
                return;
            }
            response.setContentType(TEXT_HTML_UTF8.toString());
            StringBuilder builder = new StringBuilder();
            TemporalAccessor timestamp = (TemporalAccessor) model.get("timestamp");
            String traceId = (String) model.get("traceId");
            Object message = model.get("message");
            String error = getError(model);
            Object trace = model.get("trace");
            builder.append("<html><body>");
            builder.append("<h1>").append(this.htmlEscape(message)).append("</h1>");
            if (error != null) {
                builder.append("<span class='subtitle'> 错误码: ").append(error).append("</div>");
            }

            Map<String, Object> attrs = Maps.newLinkedHashMap();
            if (timestamp != null) {
                attrs.put("Timestamp", TIME_FORMATTER.format(timestamp));
            }
            if (traceId != null) {
                attrs.put("X-TraceID", traceId);
            }
            if (!attrs.isEmpty()) {
                builder.append("<ul>");
                for (Map.Entry<String, Object> entry : attrs.entrySet()) {
                    builder.append("<li>");
                    builder.append(entry.getKey()).append(": ").append(entry.getValue());
                    builder.append("</li>");
                }
                builder.append("</ul>");
            }
            if (trace != null) {
                builder.append("<div style='white-space:pre-wrap;margin-top:10px'>").append(htmlEscape(trace)).append("</div>");
            }
            builder.append("</body></html>");
            response.getWriter().append(builder);
        }

        private String htmlEscape(Object input) {
            return (input != null) ? HtmlUtils.htmlEscape(input.toString()) : null;
        }

        private String getError(Map<String, ?> model) {
            String error = (String) model.get("error");
            if (error != null) {
                return error;
            }
            return StandardErrors.SERVICE_INTERNAL_ERROR.getErrorCode();
        }

        @Override
        public String getContentType() {
            return "text/html";
        }
    }

    @Bean(name = "error")
    public View beastErrorView() {
        return new ErrorView();
    }

    @Bean
    public BeastHandlerExceptionResolver beastHandlerExceptionResolver(ErrorMessageSource errorMessageSource) {
        BeastHandlerExceptionResolver resolver = new BeastHandlerExceptionResolver(errorMessageSource);
        resolver.setWarnLogCategory(resolver.getClass().getName());
        return resolver;
    }

    @Configuration
    public class BeastMvcConfigurer implements WebMvcConfigurer {

        private ErrorMessageSource errorMessageSource;

        public BeastMvcConfigurer(ErrorMessageSource errorMessageSource) {
            this.errorMessageSource = errorMessageSource;
        }

        @Override
        public void extendHandlerExceptionResolvers(List<HandlerExceptionResolver> resolvers) {
            //注意 删除最后一个DefaultHandlerExceptionResolver，并重写一个，升级版本可能导致错误
            resolvers.remove(resolvers.size() - 1);
            resolvers.add(new InnerHandlerExceptionResolver(this.errorMessageSource));
        }
    }


    @Bean
    public ConstraintExceptionAdvice constraintExceptionAdvice(ErrorMessageSource errorMessageSource) {
        return new ConstraintExceptionAdvice(errorMessageSource);
    }

    @ConditionalOnClass(FeignException.class)
    @Configuration
    @Slf4j
    @RestControllerAdvice
    public static class FeignExceptionAdvice {

        private ModelAndView handleException(HttpServletRequest request, HttpServletResponse response, FeignException e, String message) throws IOException {
            Request feignRequest = e.request();
            String httpMethod = null;
            String url = null;
            if (feignRequest != null) {
                httpMethod = String.valueOf(feignRequest.httpMethod());
                url = feignRequest.url();
            }
            log.warn("feign executing {} {} message:{}", httpMethod, url, message, e);
            request.setAttribute(ServerExchangeUtils.ERROR_ATTR, StandardErrors.SERVICE_INTERNAL_ERROR.getErrorCode());
            response.sendError(Response.SC_OK);
            return new ModelAndView();
        }
        @ExceptionHandler(RetryableException.class)
        public ModelAndView handleRetryableException(HttpServletRequest request, HttpServletResponse response, RetryableException e) throws IOException {
            String message = e.getMessage();
            if (e.getCause() != null) {
                message = e.getCause().getMessage();
            }
            return handleException(request, response, e, message);
        }
        @ExceptionHandler(FeignException.class)
        public ModelAndView handleFeignException(HttpServletRequest request, HttpServletResponse response, FeignException e) throws IOException {
            String message = e.getMessage();
            return handleException(request, response, e, message);
        }

    }
}
