package com.xunyi.beast.web.bind;


import com.xunyi.beast.data.message.ErrorException;
import com.xunyi.beast.data.message.IErrorOwner;
import com.xunyi.beast.data.message.Return;
import com.xunyi.beast.data.message.StandardErrors;
import com.xunyi.beast.token.web.InvalidXYTokenException;
import com.xunyi.beast.token.web.MissingXYTokenException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.UnsatisfiedServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.Iterator;
import java.util.Set;

@Slf4j
public abstract class AbstractErrorControllerAdvice {

    public abstract IErrorOwner errorOfCode(String code) throws Throwable;


    protected IErrorOwner errorOfMessage(String message, IErrorOwner defaultError) {
        try {
            return errorOfCode(message);
        } catch (Throwable e) {
            log.warn("not match error message: ", e);
            return defaultError;
        }

    }
    protected Return returnOfMessage(String message, IErrorOwner defaultError) {
        return errorOfMessage(message, defaultError).toReturn();
    }
    protected Return returnOfMessage(String message) {
       return returnOfMessage(message, StandardErrors.UNKNOWN_ERROR);
    }


    @ExceptionHandler(ConstraintViolationException.class)
    public @ResponseBody
    Return constraintViolationException(ConstraintViolationException cve) {
        Set<ConstraintViolation<?>> cvs = cve.getConstraintViolations();
        Iterator<ConstraintViolation<?>> cvIterator = cvs.iterator();
        ConstraintViolation<?> cs = cvIterator.next();
        String message = cs.getMessage();
        return errorOfMessage(message, StandardErrors.PARAMETER_BAD).toReturn(cs.getPropertyPath().toString());
    }


    protected Return returnOfBindResult(BindingResult bindingResult) {
        FieldError fieldError = bindingResult.getFieldError();
        log.warn("returnOfBindResult: Field Error{}", fieldError);
        assert fieldError != null;
        String message = fieldError.getDefaultMessage();
        return errorOfMessage(message, StandardErrors.PARAMETER_BAD).toReturn(fieldError.getField());
    }

    @ExceptionHandler({InvalidXYTokenException.class, MissingXYTokenException.class})
    public @ResponseBody Return handleInvalidXYToken(Exception e) {
        log.warn("XYTokenBindException: {}", e.getMessage(), e);
        return StandardErrors.XYTOKEN_INVALID.toReturn();
    }

    @ExceptionHandler(MethodArgumentNotValidException.class)
    public @ResponseBody Return methodArgumentNotValidException(
            HttpServletRequest request,
            MethodArgumentNotValidException e
    ) {
        log.warn("request uri:[{}] exception:", request.getRequestURI(), e);
        return returnOfBindResult(e.getBindingResult());
    }

    @ExceptionHandler(HttpMediaTypeNotAcceptableException.class)
    public @ResponseBody Return handHttpMediaTypeException(
            HttpServletRequest request,
            HttpMediaTypeNotAcceptableException e
    ) {
        log.warn("request uri:[{}] accept: [{}] supportedMediaType: [{}] exception:", request.getRequestURI(), request.getHeader("accept"), e.getSupportedMediaTypes(), e);
        return StandardErrors.HTTP_MEDIA_TYPE_NOT_ACCEPTABLE.toReturn();
    }

    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public @ResponseBody Return handHttpRequestMethodNotSupportedException(
            HttpServletRequest request,
            HttpRequestMethodNotSupportedException e
    ) {
        log.warn("request uri:[{}] method: [{}] supportedMediaType: [{}] exception:", request.getRequestURI(), e.getMethod(), e.getSupportedMethods(), e);
        return StandardErrors.HTTP_REQUEST_METHOD_NOT_SUPPORTED.toReturn();
    }

    @ExceptionHandler(MethodArgumentTypeMismatchException.class)
    public @ResponseBody Return handleMethodArgumentTypeMismatchException(
            HttpServletRequest request,
            MethodArgumentTypeMismatchException e
    ) {
        log.warn("request uri:[{}] name: {} exception:", request.getRequestURI(), e.getName(), e);
        return StandardErrors.PARAMETER_BAD.toReturn(e.getName());
    }

    @ExceptionHandler(MissingServletRequestParameterException.class)
    public @ResponseBody Return handleMissingRequestParameter(
            HttpServletRequest request,
            MissingServletRequestParameterException e
    ) {
        log.warn("request uri:[{}] name: {} exception:", request.getRequestURI(), e.getParameterName(), e);
        return StandardErrors.PARAMETER_BAD.toReturn(e.getParameterName());
    }

    @ExceptionHandler(BindException.class)
    public @ResponseBody Return bindException(
            HttpServletRequest request,
            BindException e
    ) {
        log.warn("request uri:[{}] exception:", request.getRequestURI(), e);
        return returnOfBindResult(e.getBindingResult());
    }

    @ExceptionHandler(UnsatisfiedServletRequestParameterException.class)
    public @ResponseBody Return handleUnsatisfiedServletRequestParameterException(
            HttpServletRequest request,
            UnsatisfiedServletRequestParameterException e
    ) {
        log.warn("request uri:[{}] exception: {}", request.getRequestURI(), e.getMessage(), e);
        return StandardErrors.PARAMETER_BAD.toReturn();
    }


    @ExceptionHandler(ErrorException.class)
    public @ResponseBody Return handleErrorException(ErrorException e) {
        log.warn("handle ErrorException error:{}", e.getError(), e);
        return e.toReturn();
    }

    //TODO 类未定义 异常 等处理
//    @ExceptionHandler(RetryableException.class)
//    public @ResponseBody Return handleRetryableException(RetryableException e) {
//        Request request = e.request();
//        String httpMethod = null;
//        String url = null;
//        if (request != null) {
//            httpMethod = String.valueOf(request.httpMethod());
//            url = request.url();
//        }
//        String message = null;
//        if (e.getCause() != null) {
//            message = e.getCause().getMessage();
//        }
//        log.warn("feign executing {} {} message:{}", httpMethod, url, message, e);
//        return StandardErrors.SERVICE_BUSY.toReturn();
//    }


    @ExceptionHandler(Exception.class)
    public @ResponseBody Return handleException(Exception e) {
        log.error("handleException message: {}", e.getMessage(), e);
        return StandardErrors.UNKNOWN_ERROR.toReturn();
    }

}
