package com.xunyi.beast.token.support;

import com.xunyi.beast.token.user.UserToken;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.SignatureAlgorithm;

import java.security.Key;
import java.time.Instant;
import java.util.Map;

public class UserTokenCodec implements TokenCodec<UserToken>{
    public static final String PRI_RESOURCE = "classpath:token/xy-token-private.der";
    public static final String PUB_RESOURCE = "classpath:token/xy-token-public.der";
    private static final SignatureAlgorithm SIGNATURE_ALGORITHM = SignatureAlgorithm.RS256;
    private static final class CLAIM_NAMES {
        public static final String APP = "app";
        public static final String UID = "uid";
        public static final String ISSUED_AT = "iss";
        public static final String EXPIRES_AT = "exp";
    }




    @Override
    public String encode(UserToken token) {
        return UserTokenBuilder.getInstance().compact(token);
    }

    @Override
    public UserToken decode(String tokenString) {
        return UserTokenParser.getInstance().parse(tokenString);
    }

    public static class UserTokenParser extends JwtTokenHelper.AbstractTokenJwtParser<UserToken> {

        public UserTokenParser() {
            this(JwtTokenHelper.readPublicKey(PUB_RESOURCE));
        }
        public UserTokenParser(Key key) {
            super(key);
        }

        @Override
        protected UserToken read(Claims claims) {
            UserToken token = new UserToken();
            String appId = claims.get(CLAIM_NAMES.APP, String.class);
            Long uid = claims.get(CLAIM_NAMES.UID, Long.class);
            Instant issuedAt = Instant.ofEpochMilli(claims.get(CLAIM_NAMES.ISSUED_AT, Long.class));
            Instant expiresAt = Instant.ofEpochMilli(claims.get(CLAIM_NAMES.EXPIRES_AT, Long.class));
            token.setAppId(appId);
            token.setUid(uid);
            token.setIssuedAt(issuedAt);
            token.setExpiresAt(expiresAt);
            return token;
        }

        private static final UserTokenParser INSTANCE = new UserTokenParser();

        public static UserTokenParser getInstance() {
            return INSTANCE;
        }
    }

    public static class UserTokenBuilder extends JwtTokenHelper.AbstractTokenJwtBuilder<UserToken> {

        public UserTokenBuilder() {
            this(JwtTokenHelper.readPrivateKey(PRI_RESOURCE),SIGNATURE_ALGORITHM);
        }

        public UserTokenBuilder(Key key, SignatureAlgorithm algorithm) {
            super(key, algorithm);
        }

        @Override
        protected void write(Map<String, Object> claims, UserToken token) {
            claims.put(CLAIM_NAMES.APP, token.getAppId());
            claims.put(CLAIM_NAMES.UID, token.getUid());
            claims.put(CLAIM_NAMES.ISSUED_AT, token.getIssuedAt().toEpochMilli());
            claims.put(CLAIM_NAMES.EXPIRES_AT, token.getExpiresAt().toEpochMilli());
        }

        private static final UserTokenBuilder INSTANCE = new UserTokenBuilder();

        public static UserTokenBuilder getInstance() {
            return INSTANCE;
        }
    }
}