package com.xunyi.beast.token.support;

import com.google.common.collect.Maps;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import org.apache.commons.io.IOUtils;
import org.springframework.util.ResourceUtils;

import java.net.URL;
import java.security.Key;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Map;

public abstract class JwtTokenHelper {


    public static PrivateKey readPrivateKey(String resource) {
        try {
            URL url = ResourceUtils.getURL(resource);
            byte[] keyBytes = IOUtils.toByteArray(url);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            return keyFactory.generatePrivate(keySpec);
        } catch (Exception e) {
            throw new UnsupportedOperationException(e);
        }
    }

    public static PublicKey readPublicKey(String resource) {
        try {
            URL url = ResourceUtils.getURL(resource);
            byte[] keyBytes = IOUtils.toByteArray(url);
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            return keyFactory.generatePublic(keySpec);
        } catch (Exception e) {
            throw new UnsupportedOperationException(e);
        }
    }


    public static abstract class AbstractTokenJwtParser<T> implements ITokenJwtParser<T> {
        private Key key;

        public AbstractTokenJwtParser(Key key) {
            this.key = key;
        }

        protected abstract T read(Claims claims);

        public T parse(String jwtString) {
            Jws<Claims> jws = Jwts.parser().setSigningKey(key).parseClaimsJws(jwtString);
            Claims claims = jws.getBody();
            return read(claims);
        }

    }
    public static abstract class AbstractTokenJwtBuilder<T> implements ITokenJwtBuilder<T> {
        private Key key;
        private SignatureAlgorithm algorithm;

        public AbstractTokenJwtBuilder(Key key, SignatureAlgorithm algorithm) {
            this.key = key;
            this.algorithm = algorithm;
        }

        protected abstract void write(Map<String, Object> claims, T token);

        public String compact(T token) {
            Map<String, Object> claims = Maps.newHashMap();
            this.write(claims, token);
            return Jwts.builder().signWith(key, algorithm)
                    .setClaims(claims)
                    .compact();
        }
    }
}
