package com.xunyi.beast.token.web;

import com.xunyi.beast.token.XYToken;
import com.xunyi.beast.token.support.XYTokenJwt;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.core.MethodParameter;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.annotation.MethodArgumentConversionNotSupportedException;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;
import org.springframework.web.util.UrlPathHelper;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

@Slf4j
public class XYTokenArgumentResolver implements HandlerMethodArgumentResolver {

    private UrlPathHelper urlPathHelper = new UrlPathHelper();

    private XYTokenJwt.Parser parser = new XYTokenJwt.Parser();

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return parameter.hasParameterAnnotation(XYTokenValue.class);
    }

    @Override
    public Object resolveArgument(MethodParameter parameter, @Nullable ModelAndViewContainer mavContainer,
                                  NativeWebRequest webRequest, @Nullable WebDataBinderFactory binderFactory) throws Exception {
        XYTokenValue annotation = parameter.getParameterAnnotation(XYTokenValue.class);
        Assert.state(annotation != null, "No XYTokenValue annotation");
        String name = parameter.getParameterName();
        if (name == null) {
            name = "";
        }
        MethodParameter nestedParameter = parameter.nestedIfOptional();

        Object arg = resolveValue(nestedParameter, webRequest);
        if (arg == null) {
            if (annotation.required() && !nestedParameter.isOptional()) {
                handleMissingValue(parameter);
            }
            arg = handleNullValue(arg, nestedParameter.getNestedParameterType());
        }
        if (binderFactory != null) {
            WebDataBinder binder = binderFactory.createBinder(webRequest, null, name);
            try {
                arg = binder.convertIfNecessary(arg, parameter.getParameterType(), parameter);
            }
            catch (ConversionNotSupportedException ex) {
                throw new MethodArgumentConversionNotSupportedException(arg, ex.getRequiredType(),
                        name, parameter, ex.getCause());
            }
            catch (TypeMismatchException ex) {
                throw new MethodArgumentTypeMismatchException(arg, ex.getRequiredType(),
                        name, parameter, ex.getCause());
            }
        }
        return arg;
    }

    private void handleMissingValue(MethodParameter parameter) throws MissingXYTokenException {
        throw new MissingXYTokenException(parameter);
    }

    @Nullable
    private Object handleNullValue(@Nullable Object value, Class<?> paramType) {
        if (value == null) {
            if (Boolean.TYPE.equals(paramType)) {
                return Boolean.FALSE;
            }
        }
        return value;
    }


    private @Nullable Object resolveValue(MethodParameter parameter, NativeWebRequest webRequest) throws InvalidXYTokenException {
        HttpServletRequest servletRequest = webRequest.getNativeRequest(HttpServletRequest.class);
        Assert.state(servletRequest != null, "No HttpServletRequest");
        Cookie cookieValue = WebUtils.getCookie(servletRequest, XYToken.COOKIE_NAME);
        if (cookieValue == null) {
            return null;
        }
        String xyTokenString = this.urlPathHelper.decodeRequestString(servletRequest, cookieValue.getValue());
        if (String.class.isAssignableFrom(parameter.getNestedParameterType())) {
            return xyTokenString;
        }
        else {
            try {
                return parser.parse(xyTokenString);
            } catch (Exception e) {
                log.warn("token [{}] parse  exception", xyTokenString, e);
                throw new InvalidXYTokenException(parameter);
            }
        }
    }

//    @Override
//    public Object resolveArgument(MethodParameter parameter, ModelAndViewContainer mavContainer,
//              NativeWebRequest webRequest, WebDataBinderFactory binderFactory) throws Exception {
////        webRequest.
//        webRequest.getParameter()
//        return null;
//    }
}
