package com.xunyi.beast.payment.core;

import com.google.common.base.Strings;
import com.google.common.hash.Hashing;
import lombok.extern.slf4j.Slf4j;


/**
 * 尽量确保当天ID尽量唯一
 * 外部调用需要 重复判断，重试处理
 */
@Slf4j
public class OneDayIdGenerator {

    private final static long TIMESTAMP_BIT = 26;
    private final static long DATA_CENTER_BIT = 5;
    private final static long WORKER_BIT = 5;
    private final static long SEQUENCE_BIT = 10;

    private final static long TIMESTAMP_MASK = ~(-1L << TIMESTAMP_BIT);
    private final static long DATA_CENTER_MASK = ~(-1L << DATA_CENTER_BIT);
    private final static long WORKER_MASK = ~(-1L << WORKER_BIT);
    private final static long SEQUENCE_MASK = ~(-1L << SEQUENCE_BIT);

    private final static long TIMESTAMP_SHIFT = DATA_CENTER_BIT + SEQUENCE_BIT + WORKER_BIT;
    private final static long DATA_CENTER_SHIFT = SEQUENCE_BIT + WORKER_BIT;
    private final static long WORKER_SHIFT = SEQUENCE_BIT;


    private long lastTime = -1L;
    private long dataCenterId;
    private long workerId;
    private long sequence = 0L;

    public OneDayIdGenerator(long dataCenterId, long workerId) {
        this.dataCenterId = dataCenterId;
        this.workerId = workerId;
    }

    protected long time() {
        return System.currentTimeMillis();
    }

    protected long nextTime() {
        long timestamp = time();
        while (timestamp <= lastTime) {
            timestamp = time();
        }
        return timestamp;
    }

    /**
     * 基于当天趋近唯一
     */
    private synchronized long nextIncId() {
        long now = this.time();
        if (now < lastTime) {
            throw new RuntimeException("Clock moved backwards.  Refusing to generate id");
        }
        if (now == lastTime) {
            sequence = (sequence + 1) & SEQUENCE_MASK;
            if (sequence == 0L) {
                now = nextTime();
            }
        } else {
            sequence = 0L;
        }
        lastTime = now;

        //86400000
        long time = now % (24 * 3600 * 1000);

        return (time) << TIMESTAMP_SHIFT   //时间
                | this.dataCenterId << DATA_CENTER_SHIFT  //
                | this.workerId << WORKER_SHIFT  //
                | this.sequence;
    }

    /**
     * 生成9为分布式ID （趋近于当天唯一性）
     */
    public synchronized String nextId() {
        long nextId= this.nextIncId();
        int id = Hashing.murmur3_32().hashLong(nextId).asInt();
        if (id < 0) {
            id = -id;
        }
        id = id % (1000000000);
        return Strings.padStart(String.valueOf(id), 9, '0');
    }

//    public String nextTradeNo() {
//        LocalDate now = LocalDate.now();
//        String date = formatter.format(now);
//        String no = nextNo();
//        return date + no;
//        int retry = 3;
//        String date = null;
//        do {
//            //前面使用年月日，也是当作版本号作用，保证生成算法可切换
//            LocalDate now = LocalDate.now();
//
//            date = formatter.format(now);
//
//            String no = nextNo();
//            String tradeNo = date + no;
//
//            if (tradeNoRepository.add(now, tradeNo)) {
//                return tradeNo;
//            } else {
//                log.warn("生成碰撞{} ,trade no:{}", now, tradeNo);
//            }
//        } while (retry -- > 0);
//        throw new IllegalStateException("generator next No failed date:" + date);
//    }

}
