package com.xunyi.beast.payment.channel.wechat;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.codec.digest.HmacAlgorithms;
import org.apache.commons.codec.digest.HmacUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
public class WechatPayUtils {

    private static final String FIELD_SIGN = "sign";

    private static DocumentBuilder newDocumentBuilder() throws ParserConfigurationException {
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
        documentBuilderFactory.setFeature("http://xml.org/sax/features/external-general-entities", false);
        documentBuilderFactory.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
        documentBuilderFactory.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
        documentBuilderFactory.setFeature(XMLConstants.FEATURE_SECURE_PROCESSING, true);
        documentBuilderFactory.setXIncludeAware(false);
        documentBuilderFactory.setExpandEntityReferences(false);

        return documentBuilderFactory.newDocumentBuilder();
    }

    private static Document newDocument() throws ParserConfigurationException {
        return newDocumentBuilder().newDocument();
    }

    /**
     * XML格式字符串转换为Map
     *
     * @param strXML XML字符串
     * @return XML数据转换后的Map
     * @throws Exception
     */
    public static Map<String, String> xmlToMap(String strXML) throws Exception {
        try {
            Map<String, String> data = new HashMap<String, String>();
            DocumentBuilder documentBuilder = WechatPayUtils.newDocumentBuilder();
            InputStream stream = new ByteArrayInputStream(strXML.getBytes("UTF-8"));
            Document doc = documentBuilder.parse(stream);
            doc.getDocumentElement().normalize();
            NodeList nodeList = doc.getDocumentElement().getChildNodes();
            for (int idx = 0; idx < nodeList.getLength(); ++idx) {
                Node node = nodeList.item(idx);
                if (node.getNodeType() == Node.ELEMENT_NODE) {
                    org.w3c.dom.Element element = (org.w3c.dom.Element) node;
                    data.put(element.getNodeName(), element.getTextContent());
                }
            }
            try {
                stream.close();
            } catch (Exception ex) {
                // do nothing
            }
            return data;
        } catch (Exception ex) {
            WechatPayUtils.getLogger().warn("Invalid XML, can not convert to map. Error message: {}. XML content: {}", ex.getMessage(), strXML);
            throw ex;
        }

    }


    public static boolean isSignatureValid(Map<String, String> params, String key, WechatSignType signType) {
        if (!params.containsKey(FIELD_SIGN)) {
            log.warn("sign is empty");
            return false;
        }
        Map<String, String> contentParams = Maps.newHashMap(params);
        String sign = contentParams.remove(FIELD_SIGN);
        return generateSignature(contentParams, key, signType).equalsIgnoreCase(sign);

    }
    /**
     *
     * @param params
     * @param key
     * @param signType
     * @return
     */
    public static String generateSignature(Map<String, String> params, String key, WechatSignType signType) {
        List<String> items = params.entrySet().stream().sorted(new Comparator<Map.Entry<String, String>>() {
            @Override
            public int compare(Map.Entry<String, String> o1, Map.Entry<String, String> o2) {
                return o1.getKey().compareTo(o2.getKey());
            }
        }).filter(item -> {
            return !Strings.isNullOrEmpty(item.getValue());
        }).map(item -> {
            return item.getKey() + "=" + item.getValue();
        }).collect(Collectors.toList());
        String content = Joiner.on("&").join(items) + "&key=" + key;
        switch (signType) {
            case MD5: return DigestUtils.md5Hex(content);
            case HMACSHA256: return new HmacUtils(HmacAlgorithms.HMAC_SHA_256, key).hmacHex(content);
            default: throw new IllegalArgumentException(String.format("Invalid sing type: %s", signType));
        }
    }


    public static Logger getLogger() {
        return LoggerFactory.getLogger("wxpay java sdk");
    }


    public static String returnNotifySuccess() {
        return "<xml><return_code><![CDATA[SUCCESS]]></return_code><return_msg><![CDATA[OK]]></return_msg></xml>";
    }
    public static String returnNotifyFailure(String msg) {
        return "<xml><return_code><![CDATA[FAILURE]]></return_code><return_msg><![CDATA[" + msg + "]]></return_msg></xml>";
    }

}
