package com.xunlei.niux.data.vipgame.bo.activity;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import javax.annotation.Resource;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Service;

import com.ferret.common.dao.BaseDao;
import com.xunlei.niux.data.vipgame.proxy.JRedisProxy;
import com.xunlei.niux.data.vipgame.vo.activity.VirtualCard;

/**
 * 虚拟卡。目前支持积分双倍卡和积分半价卡
 * @author 陆建明
 *
 */
@Service()
public class VirtualCardBoImpl implements VirtualCardBo {
    private static Logger logger = Logger.getLogger(VirtualCardBoImpl.class);
	protected static SimpleDateFormat sdf_time = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	private static final int day = 1;//激活后有效使用天数
	
	@Resource(name="BaseDao")
	private BaseDao baseDao;

	public BaseDao getBaseDao() {
		return baseDao;
	}

	public void setBaseDao(BaseDao baseDao) {
		this.baseDao = baseDao;
	}
	
	/**
	 * 发放虚拟卡
	 * 
	 * @param type 卡类型 1:双倍积分卡 2:半价卡 不能为空
	 * @param userId 用户迅雷UID  不能为空
	 * @param orderId 订单号 不能为空
	 * @param orderActNo 领取虚拟卡时的活动编号  不能为空
	 * @return VirtualCard 返回新增的订单 为空表示本次未发放成功
	 * 
	 */
	@Override
	public VirtualCard giveOut(String type, String userId, String orderId, String orderActNo, String ip) {
		
		//判断参数
		if(StringUtils.isEmpty(type) || StringUtils.isEmpty(userId) || StringUtils.isEmpty(orderId)){
			return null;
		}
		
		//查询订单是否重复
		String sql = "select * from virtualcard where orderId = ? limit 1";
		List<Object> paramList = new ArrayList<Object>();
		paramList.add(orderId);
		
		List<VirtualCard> virtualCards = baseDao.executeQuery(VirtualCard.class, sql, paramList);//baseDao.findBySql(VirtualCard.class, sql);
		if(virtualCards != null && virtualCards.size() > 0 ){
			//返回空，表示本次未发放成功
			return null;
		}
		
		//生成新的订单
		VirtualCard virtualCard = new VirtualCard();
		virtualCard.setType(type);
		virtualCard.setStatus(2);//已领取
		virtualCard.setUserId(userId);
		virtualCard.setOrderId(orderId);
		virtualCard.setOrderTime(sdf_time.format(new Date()));
		virtualCard.setOrderActNo(orderActNo);
		virtualCard.setIp(ip);
		
		baseDao.insert(virtualCard);
		
        return virtualCard;
	}


	/**
	 * 激活虚拟卡。同一时间不能同时使用多张同一类型卡。
	 * 
	 * @param type 卡类型 1:双倍积分卡 2:半价卡 不能为空
	 * @param userId 用户迅雷UID  不能为空
	 * @param orderActNo 领取虚拟卡时的活动编号  不能为空
	 * @param orderId 订单号  可以为空，为空时随便选择一张卡来激活
	 * @return VirtualCard 返回新增的订单 为空表示本次未发放成功
	 * 
	 */
	@Override
	public int use(String type, String userId, String actNo, String orderId) {
		
		//判断参数
		if(StringUtils.isEmpty(type) || StringUtils.isEmpty(userId)){
			return 2;
		}

        //1.先从缓存取是否有激活卡号KEY：（"VirtualCard_" + userId + "_" + type）
        String expireTime = null;
        try{
        	expireTime = JRedisProxy.getInstance().get("VirtualCard_" + userId + "_" + type);
        }catch(Exception e){
        	logger.warn("use error for cache reading. code=1,type=" + type + ",userId="+ userId + ",actNo=" + actNo, e);
        }
        
        if(StringUtils.isEmpty(expireTime)){
    		//1.1查询已激活的卡
    		List<VirtualCard> virtualCards =  getUsedVirtualCard(type, userId);
    		if(virtualCards != null && virtualCards.size() > 0 ){
    			//返回1，表示已有已激活的有效虚拟卡
    			return 1;
    		}
        }else if(expireTime.compareTo(sdf_time.format(new Date())) > 0){
        	logger.warn("use error for cache conflict. code=1,type=" + type + ",userId="+ userId + ",actNo=" + actNo+",expireTime="+expireTime);
            return 1;
        }
		
		//2.查询有可激活的卡
		List<VirtualCard> virtualCards_unused = getUnusedVirtualCard(type, userId, orderId);
		
		//2.1没有可使用的卡
		if(virtualCards_unused == null || virtualCards_unused.isEmpty()){
			//返回3，表示当前没有已激活的有效虚拟卡
			return 3;
		}
		
		//2.2有可使用的卡
		VirtualCard virtualCard = virtualCards_unused.get(0);
		virtualCard.setStatus(4);//已激活
		virtualCard.setActNo(actNo);
		virtualCard.setBeginTime(sdf_time.format(new Date()));
    	int cacheExpireTime = day * 24 * 60 * 60;
		String endTime = sdf_time.format(new Date(System.currentTimeMillis() + cacheExpireTime * 1000L));
		virtualCard.setEndTime(endTime);
		
		baseDao.updateById(virtualCard);

    	//设置缓存
		boolean updateCache = false;
		try{
			updateCache = JRedisProxy.getInstance().set("VirtualCard_" + userId + "_" + type, endTime, cacheExpireTime);
		}catch(Exception e){
	    	logger.warn("use error for cache reading. code=1,type=" + type + ",userId="+ userId + ",actNo=" + actNo + ",endTime=" +endTime + ",cacheExpireTime="+cacheExpireTime+",updateCache=" + updateCache, e);
	    }
		
        return 0;
	}


	/**
	 * 查询用户是否有虚拟卡。
	 * 
	 * @param type 卡类型 0:双倍积分卡 1:半价卡 不能为空
	 * @param userId 用户迅雷UID  不能为空
	 * @return boolean true:当前用户有已激活卡 false：当前用户无已激活卡
	 * 
	 */
	@Override
	public boolean check(String type, String userId) {
		
		//判断参数
		if(StringUtils.isEmpty(type) || StringUtils.isEmpty(userId)){
			return false;
		}

        //1.先从缓存取是否有激活卡号KEY：（"VirtualCard_" + userId + "_" + type）
        String expireTime = null;
        try{
        	expireTime = JRedisProxy.getInstance().get("VirtualCard_" + userId + "_" + type);
        }catch(Exception e){
        	logger.warn("use error for cache reading. code=1,type=" + type + ",userId="+ userId, e);
        }
        
        if(StringUtils.isEmpty(expireTime)){
    		//1.1查询已激活的卡
    		List<VirtualCard> virtualCards = getUsedVirtualCard(type, userId);
    		if(virtualCards != null && virtualCards.size() > 0 ){
    			//返回1，表示已有已激活的有效虚拟卡
    			return true;
    		}
        }else if(expireTime.compareTo(sdf_time.format(new Date())) > 0){
        	logger.warn("use error for cache conflict. code=1,type=" + type + ",userId="+ userId +",expireTime="+expireTime);
            return true;
        }
        
        return false;
	}
	
	/**
	 * 获取当前已经激活的虚拟卡
	 * @param type String
	 * @param userId String
	 * @return virtualCards List<VirtualCard>
	 */
	private List<VirtualCard> getUsedVirtualCard(String type, String userId){

		//1.1查询是否已有已激活的卡
		String sql = "select * from virtualcard where type = ? and userId = ? and status = ? and beginTime is not null and beginTime <= ? and endTime is not null and endTime >= ? limit 1";
		List<Object> paramList = new ArrayList<Object>();
		paramList.add(type);
		paramList.add(userId);
		paramList.add(4);//已激活
		String nowTime = sdf_time.format(new Date());
		paramList.add(nowTime);
		paramList.add(nowTime);
		
		//已经有激活的卡，则直接返回此卡
		List<VirtualCard> virtualCards = baseDao.executeQuery(VirtualCard.class, sql, paramList);
		
		return virtualCards;
	}
	
	/**
	 * 获取可使用的虚拟卡
	 * @param type String
	 * @param userId String
	 * @param orderId String 
	 * @return virtualCards List<VirtualCard>
	 */
	private List<VirtualCard> getUnusedVirtualCard(String type, String userId, String orderId){
		String sql_unused = "select * from virtualcard where type = ? and userId = ? and status = ? and (expireTime is null or expireTime >= ?)";
		List<Object> paramList_unused = new ArrayList<Object>();
		paramList_unused.add(type);
		paramList_unused.add(userId);
		paramList_unused.add(2);//已领取（未激活）
		String nowTime_unused = sdf_time.format(new Date());
		paramList_unused.add(nowTime_unused);
		
		if(!StringUtils.isEmpty(orderId)){
			sql_unused += " and orderId = ? ";
			paramList_unused.add(orderId);
		}
		
		sql_unused += " limit 1 for update";
		
		List<VirtualCard> virtualCards_unused = baseDao.executeQuery(VirtualCard.class, sql_unused, paramList_unused);
		
		return virtualCards_unused;
	}

	@Override
	public boolean invalid(String type, String userId, String actNo) {
		
		//判断参数
		if(StringUtils.isEmpty(type) || StringUtils.isEmpty(userId)){
			return false;
		}

        //1.先清空缓中的激活卡KEY：（"VirtualCard_" + userId + "_" + type）
        try{
        	JRedisProxy.getInstance().del("VirtualCard_" + userId + "_" + type);
        }catch(Exception e){
        	logger.warn("use error for cache del. code=1,type=" + type + ",userId="+ userId, e);
        }
        
		//1.1修改已激活的卡的状态
		List<VirtualCard> virtualCards = getUsedVirtualCard(type, userId);
		if(virtualCards != null && virtualCards.size() > 0 ){
			//2.2有可使用的卡,一次使一张失效
			VirtualCard virtualCard = virtualCards.get(0);
			virtualCard.setStatus(3);//已失效
			virtualCard.setActNo(actNo);
			virtualCard.setExpireTime(sdf_time.format(new Date()));
			
			baseDao.updateById(virtualCard);
			
			//返回1，表示已有已激活的有效虚拟卡
			return true;
		}
        
        return false;
	}
	
}
