package com.xunyi.env;


import com.google.common.collect.Maps;
import com.xunyi.utils.PropertyPlaceholderHelper;
import com.xunyi.utils.SystemPropertyUtils;

import java.util.Map;

public class PropertySourcesPropertyResolver {

    private final MutablePropertySources propertySources;

    private String placeholderPrefix = SystemPropertyUtils.PLACEHOLDER_PREFIX;

    private String placeholderSuffix = SystemPropertyUtils.PLACEHOLDER_SUFFIX;

    private String valueSeparator = SystemPropertyUtils.VALUE_SEPARATOR;


    private PropertyPlaceholderHelper nonStrictHelper;
    private PropertyPlaceholderHelper strictHelper;

    private Map<String, String> standardProperties = Maps.newHashMap();

    private static String[] keys = new String[]{"env", "domain"};
    public PropertySourcesPropertyResolver(MutablePropertySources propertySources) {
        this.propertySources = propertySources;
        for (String key : keys) {
            this.standardProperties.put(key, this.getProperty(key));
        }

    }


    public boolean containsProperty(String key) {
        if (this.propertySources != null) {
            for (PropertySource<?> propertySource : this.propertySources) {
                if (propertySource.containsProperty(key)) {
                    return true;
                }
            }
        }
        return false;
    }

    public String getProperty(String key) {
        return getProperty(key, String.class, true);
    }

    public String getProperty(String key, String defaultValue) {
        String value = getProperty(key);
        return (value != null ? value : defaultValue);
    }

    public <T> T getProperty(String key, Class<T> targetValueType) {
        return getProperty(key, targetValueType, true);
    }

    public <T> T getProperty(String key, Class<T> targetType, T defaultValue) {
        T value = getProperty(key, targetType);
        return (value != null ? value : defaultValue);
    }
    public String getRequiredProperty(String key) throws IllegalStateException {
        String value = getProperty(key);
        if (value == null) {
            throw new IllegalStateException("Required key '" + key + "' not found");
        }
        return value;
    }

    public <T> T getRequiredProperty(String key, Class<T> valueType) throws IllegalStateException {
        T value = getProperty(key, valueType);
        if (value == null) {
            throw new IllegalStateException("Required key '" + key + "' not found");
        }
        return value;
    }


    protected <T> T getProperty(String key, Class<T> targetValueType, boolean resolveNestedPlaceholders) {
        if (this.propertySources != null) {
            for (PropertySource<?> propertySource : this.propertySources) {
                Object value = propertySource.getProperty(key);
                if (value != null) {
                    //临时方案
                    if (targetValueType.isAssignableFrom(String.class) && !standardProperties.containsKey(key)) {
                        value = resolvePlaceholders((String) value);
                    }

                    //简单处理下 有需要再扩展
                    if (targetValueType.isAssignableFrom(value.getClass())) {
                        return (T)value;
                    }

                    if (targetValueType == Integer.class && value instanceof String) {
                        return (T) Integer.valueOf((String) value);
                    }


                    throw new RuntimeException(String.format("value type: %s target type: %s", value.getClass().getName(), targetValueType.getName()));
                }

            }
        }
        return null;
    }


    private PropertyPlaceholderHelper createPlaceholderHelper(boolean ignoreUnresolvablePlaceholders) {
        return new PropertyPlaceholderHelper(this.placeholderPrefix, this.placeholderSuffix,
                this.valueSeparator, ignoreUnresolvablePlaceholders);
    }

    public String resolvePlaceholders(String text) {
        if (this.nonStrictHelper == null) {
            this.nonStrictHelper = createPlaceholderHelper(true);
        }
        return doResolvePlaceholders(text, this.nonStrictHelper);
    }

    public String resolveRequiredPlaceholders(String text) throws IllegalArgumentException {
        if (this.strictHelper == null) {
            this.strictHelper = createPlaceholderHelper(false);
        }
        return doResolvePlaceholders(text, this.strictHelper);
    }

    private String doResolvePlaceholders(String text, PropertyPlaceholderHelper helper) {
        return helper.replacePlaceholders(text, new PropertyPlaceholderHelper.PlaceholderResolver() {
            @Override
            public String resolvePlaceholder(String placeholderName) {
                return getPropertyAsRawString(placeholderName);
            }
        });
    }

    protected String getPropertyAsRawString(String key) {
        return getProperty(key, String.class, false);
    }

}
