package com.xunlei.netty.soaserver.cmd;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.jboss.netty.handler.codec.http.HttpHeaders;
import org.jboss.netty.handler.codec.http.HttpMethod;
import org.jboss.netty.handler.codec.http.HttpVersion;
import org.slf4j.Logger;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.xunlei.common.util.XLRuntimeException;
import com.xunlei.netty.httpserver.cmd.BaseCmd;
import com.xunlei.netty.httpserver.component.XLHttpRequest;
import com.xunlei.netty.httpserver.component.XLHttpResponse;
import com.xunlei.netty.httpserver.model.CookieInfo;
import com.xunlei.netty.soaserver.component.GenericBase;
import com.xunlei.netty.soaserver.component.SOAListResponse;
import com.xunlei.netty.soaserver.component.SOAPageResponse;
import com.xunlei.netty.soaserver.component.SOAResponse;
import com.xunlei.netty.util.SignUtil;
import com.xunlei.util.Log;

/**
 * SOA接口抽象基类
 * 
 * @author wangcanyi
 *
 */
public abstract class BaseSOACmd {
	private static Logger logger = Log.getLogger(BaseSOACmd.class.getName());

	/**
	 * Http接口转SOA接口
	 * 
	 * @param classType 返回值类型
	 * @param httpCmd HttpCmd实体
	 * @param methodName HttpCmd方法名
	 * @param param Http Url参数
	 * @return
	 * @throws Exception
	 */
	protected <T> T getSOAByHTTP(Class<T> classType, BaseCmd httpCmd, String methodName, Map<String, String> param) throws Exception {
		return getSOAByHTTPWithCookie(classType, httpCmd, methodName, param, null);
	}
	
	/**
	 * Http接口转SOA接口
	 * 
	 * @param classType 返回值类型
	 * @param httpCmd HttpCmd实体
	 * @param methodName HttpCmd方法名
	 * @param param Http Url参数
	 * @param cookieInfo cookie信息
	 * @return
	 * @throws Exception
	 */
	protected <T> T getSOAByHTTPWithCookie(Class<T> classType, BaseCmd httpCmd, String methodName, Map<String, String> param, CookieInfo cookieInfo) throws Exception {
		T resultObject = null;
		Object resultMsg = this.getHTTPMethodResultWithCookie(httpCmd, methodName, param, cookieInfo);
		if (resultMsg != null) {
			// 处理JSONP
			String resultMsgStr = resultMsg.toString().trim();
			if(!resultMsgStr.startsWith("{")) {
				resultMsgStr = resultMsgStr.substring(resultMsgStr.indexOf("{"), resultMsgStr.length() - 1);
			}
			JsonObject jsonObject = new JsonParser().parse(resultMsgStr).getAsJsonObject();
			int code = jsonObject.get("rtn").getAsInt();
			if (code == 0) {
				if (classType != null) {
					Gson gson = new Gson();
					resultObject = gson.fromJson(jsonObject.get("data"), classType);
				}
			} else {
				String msg = jsonObject.get("data").getAsString();
				throw new XLRuntimeException(msg);
			}
		} else {
			throw new XLRuntimeException("resultMsg为空");
		}
		return resultObject;
	}

	/**
	 * Http接口转SOA响应接口
	 * 
	 * @param classType SOAResponse里data类型
	 * @param httpCmd HttpCmd实体
	 * @param methodName HttpCmd方法名
	 * @param param Http Url参数
	 * @return
	 * @throws Exception
	 */
	protected <T> SOAResponse<T> getSOAResponseByHTTP(Class<T> classType, BaseCmd httpCmd, String methodName, Map<String, String> param) throws Exception {
		SOAResponse<T> resultObject = null;
		Object resultMsg = this.getHTTPMethodResult(httpCmd, methodName, param);
		if (resultMsg != null) {
			resultObject = new SOAResponse<T>(classType);
			JsonObject jsonObject = new JsonParser().parse(resultMsg.toString()).getAsJsonObject();
			int code = jsonObject.get("rtn").getAsInt();
			resultObject.setRtn(code);
			if (code == 0) {
				if (classType != null) {
					Gson gson = new Gson();
					resultObject.setData(gson.fromJson(jsonObject.get("data"), classType));
				}
			} else {
				String msg = jsonObject.get("data").getAsString();
				resultObject.setMsg(msg);
			}
		}
		return resultObject;
	}
	
	/**
	 * Http接口转SOA列表响应接口
	 * 
	 * @param classType SOAResponse里data类型
	 * @param httpCmd HttpCmd实体
	 * @param methodName HttpCmd方法名
	 * @param param Http Url参数
	 * @return
	 * @throws Exception
	 */
	@SuppressWarnings("unchecked")
	protected <T> SOAListResponse<T> getSOAListResponseByHTTP(Class<T> classType, BaseCmd httpCmd, String methodName, Map<String, String> param)
			throws Exception {
		SOAListResponse<T> resultObject = null;
		Object resultMsg = this.getHTTPMethodResult(httpCmd, methodName, param);
		if (resultMsg != null) {
			resultObject = new SOAListResponse<T>(classType);
			JsonObject jsonObject = new JsonParser().parse(resultMsg.toString()).getAsJsonObject();
			int code = jsonObject.get("rtn").getAsInt();
			resultObject.setRtn(code);
			if (code == 0) {
				if (classType != null) {
					Gson gson = new Gson();
					JsonObject jsonPages = jsonObject.get("data").getAsJsonObject();
					Type dataListType = GenericBase.getType(List.class, classType);
					resultObject.setDataList((List<T>) gson.fromJson(jsonPages, dataListType));
				}
			} else {
				String msg = jsonObject.get("data").getAsString();
				resultObject.setMsg(msg);
			}
		}
		return resultObject;
	}

	/**
	 * Http接口转SOA分页响应接口,分页数据Key(pageDatas),总数Key(totalCount)
	 * 
	 * @param classType SOAResponse里data类型
	 * @param httpCmd HttpCmd实体
	 * @param methodName HttpCmd方法名
	 * @param param Http Url参数
	 * @return
	 * @throws Exception
	 */
	protected <T> SOAPageResponse<T> getSOAPageResponseByHTTP(Class<T> classType, BaseCmd httpCmd, String methodName, Map<String, String> param)
			throws Exception {
		return getSOAPageResponseByHTTP(classType, httpCmd, methodName, param, null, null);
	}

	/**
	 * Http接口转SOA分页响应接口
	 * 
	 * @param classType SOAResponse里data类型
	 * @param httpCmd HttpCmd实体
	 * @param methodName HttpCmd方法名
	 * @param param Http Url参数
	 * @param pageDatasKey 分页数据Key
	 * @param totalCountKey 总数Key
	 * @return
	 * @throws Exception
	 */
	@SuppressWarnings("unchecked")
	protected <T> SOAPageResponse<T> getSOAPageResponseByHTTP(Class<T> classType, BaseCmd httpCmd, String methodName, Map<String, String> param,
			String pageDatasKey, String totalCountKey) throws Exception {
		SOAPageResponse<T> resultObject = null;
		Object resultMsg = this.getHTTPMethodResult(httpCmd, methodName, param);
		if (resultMsg != null) {
			resultObject = new SOAPageResponse<T>(classType);
			JsonObject jsonObject = new JsonParser().parse(resultMsg.toString()).getAsJsonObject();
			int code = jsonObject.get("rtn").getAsInt();
			resultObject.setRtn(code);
			if (code == 0) {
				if (classType != null) {
					JsonObject jsonPages = jsonObject.get("data").getAsJsonObject();
					if (StringUtils.isBlank(pageDatasKey))
						pageDatasKey = "pageDatas";
					if (StringUtils.isBlank(totalCountKey))
						totalCountKey = "totalCount";
					Gson gson = new Gson();
					resultObject.setTotalCount(jsonPages.get(totalCountKey).getAsInt());
					Type pageDatasType = GenericBase.getType(List.class, classType);
					resultObject.setPageDatas((List<T>) gson.fromJson(jsonPages.get(pageDatasKey), pageDatasType));
				}
			} else {
				String msg = jsonObject.get("data").getAsString();
				resultObject.setMsg(msg);
			}
		}
		return resultObject;
	}

	/**
	 * 获取HTTP方法的返回值
	 * 
	 * @param httpCmd
	 * @param methodName
	 * @param param
	 * @return
	 * @throws Exception
	 */
	protected Object getHTTPMethodResult(BaseCmd httpCmd, String methodName, Map<String, String> param) throws Exception {
		return getHTTPMethodResultWithCookie(httpCmd, methodName, param, null);
	}
	
	/**
	 * 获取HTTP方法的返回值(携带cookie信息)
	 * 
	 * @param httpCmd
	 * @param methodName
	 * @param param
	 * @param cookieInfo
	 * @return
	 * @throws Exception
	 */
	protected Object getHTTPMethodResultWithCookie(BaseCmd httpCmd, String methodName, Map<String, String> param, CookieInfo cookieInfo) throws Exception {
		if (StringUtils.isBlank(methodName))
			throw new XLRuntimeException("请求方法为空");
		if (httpCmd == null)
			throw new XLRuntimeException("请求对象为空");
		String uri = "?";
		if (param != null && param.size() > 0) {
			String paramStr = SignUtil.getSignatureContent(param);
			uri += paramStr;
		}
		Method method = httpCmd.getClass().getMethod(methodName, XLHttpRequest.class, XLHttpResponse.class);
		if (method == null)
			throw new XLRuntimeException("方法不存在 methodName：" + methodName);
		logger.debug("Http接口转SOA接口[getSOAByHTTP] uri:" + uri);
		XLHttpRequest request = new XLHttpRequest(HttpVersion.HTTP_1_0, HttpMethod.GET, uri);
		request.setHeader("x-forwarded-for", "127.0.0.1");
		request.setHeader("X-real-IP", "127.0.0.1");
		
		if(cookieInfo != null) {
			request.setHeader(HttpHeaders.Names.COOKIE, cookieInfo.toString());
		}
		
		XLHttpResponse response = new XLHttpResponse();
		Object resultMsg = method.invoke(httpCmd, request, response);
		logger.debug("Http接口转SOA接口[getSOAByHTTP] resultMsg:" + resultMsg);
		return resultMsg;
	}

	/**
	 * 对象按属性转成Map。复杂类嵌套未做处理
	 * 
	 * @param obj
	 * @return
	 * @throws Exception
	 */
	protected Map<String, String> getMapByObject(Object obj) throws Exception {
		Map<String, String> result = new HashMap<String, String>();
		if (obj != null) {
			Class<?> classType = obj.getClass();
			Field[] fieldArray = classType.getDeclaredFields();
			if (fieldArray != null && fieldArray.length > 0) {
				for (Field field : fieldArray) {
					field.setAccessible(true);
					String key = field.getName();
					Object value = field.get(obj);
					if (value != null)
						result.put(key, String.valueOf(value));
				}
			}
		}
		return result;
	}
}
