/**
 * 
 */
package com.xunlei.netty.soaserver.client;

import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.net.InetSocketAddress;
import java.net.Socket;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;

import com.google.gson.Gson;
import com.xunlei.netty.consul.ConsulFactory;
import com.xunlei.netty.exception.BusinessRuntimeException;
import com.xunlei.netty.soaserver.component.RpcInput;
import com.xunlei.netty.soaserver.component.RpcOutput;
import com.xunlei.netty.soaserver.component.SOAServiceBase;
import com.xunlei.netty.soaserver.exception.SOAServerRuntimeException;
import com.xunlei.netty.util.HttpUtil;
import com.xunlei.netty.util.Log;
import com.dianping.cat.Cat;
import com.dianping.cat.CatConstants;
import com.dianping.cat.message.Event;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.spi.MessageTree;
import com.ecwid.consul.v1.catalog.model.CatalogService;
import com.ecwid.consul.v1.health.model.HealthService;

/**
 * 接口动态代理基类
 * 
 * @author wangcanyi
 *
 */
@SuppressWarnings("rawtypes")
public class InterfaceProxyBase implements InvocationHandler {

	private static final Logger log = Log.getLogger();

	private Class serviceInterface;// 服务接口
	private SOAServiceBase serviceBase;

	/**
	 * JSON序列化
	 */
	private Gson gson = new Gson();
	private final String charset_utf8 = "UTF-8";

	/**
	 * 构造器
	 * 
	 * @param serviceBase 服务基类
	 */
	public InterfaceProxyBase(SOAServiceBase serviceBase, Class serviceInterface) {
		this.serviceBase = serviceBase;
		this.serviceInterface = serviceInterface;
	}

	@Override
	public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
		String methodName = method.getName();
		if (methodName.equals("equals") || methodName.equals("hashCode") | methodName.equals("toString")) {// 特殊方法处理
			/**
			 * “http://blog.csdn.net/u014236541/article/details/49338351” equals()、hashcode()、toString()此三个方法会进行动态代理方法。
			 * 动态代理类不仅代理了显示定义的接口中的方法，而且还代理了java的根类Object中的继承而来的equals()、hashcode()、toString()这三个方法，并且仅此三个方法
			 * 
			 * “http://blog.csdn.net/q99261581/article/details/49944003” invoke列循环： 在invoke（）函数中使用proxy调用方法会陷入死循环 invoke（）中proxy.getClass()并不会触发动态代理。
			 * 它输出的是接口的Class对象，而不是被代理类的Class对象。
			 */
			Object[] newArgs = null;
			if (args != null && args.length > 0) {
				newArgs = new Object[args.length];
				for (int i = 0; i < args.length; i++) {
					Object object = args[i];
					if (Proxy.isProxyClass(object.getClass())) {// 若为动态代理类，需要getClass()获取真实的类
						newArgs[i] = object.getClass();
					} else {
						newArgs[i] = object;
					}
				}
			}
			return method.invoke(proxy.getClass(), newArgs);
		} else {// 进行SOA接口调用
			log.debug(String.format("Netty框架.SOA.接口调用[InterfaceProxyBase.invoke],proxy:%s,method:%s,args:%s", proxy.getClass().getName(), method.getName(),
					gson.toJson(args)));
			RpcOutput rpcOutput = new RpcOutput();
			rpcOutput.setRpcObject(serviceInterface, method);// 设置接口方法信息
			rpcOutput.setMethodObjectArgs(args, this.serviceBase.getSignKey());
			return invoke(rpcOutput);
		}
	}

	/**
	 * 调用接口
	 * 
	 * @param rpcOutput rpcOutput
	 * @return
	 * @throws Exception
	 */
	public Object invoke(RpcOutput rpcOutput) throws Exception {
		Object resultObj = null;
		log.debug(String.format("Netty框架.SOA.接口调用[InterfaceProxyBase.invoke],rpcOutput:%s", gson.toJson(rpcOutput)));
		rpcOutput.setIpAddress(HttpUtil.getLocalSampleIP());
		rpcOutput.setAppName(Cat.getManager().getDomain());
		Transaction t = Cat.newTransaction("PigeonCall", rpcOutput.getSimpleName());
		try {
			RpcInput rpcInput = this.sendRequestRetry(rpcOutput);
			if (rpcInput != null) {
				// 验证Token，自己发起的请求，验证返回的是否是自己的数据
				if (StringUtils.isNotBlank(rpcInput.getToken()) && rpcInput.getToken().equals(rpcOutput.getToken())) {
					if (rpcInput.getError() == null || rpcInput.getError().getException() == null) {// 无异常时返回值
						if (rpcInput.getIsSuccess().booleanValue()) {// 是否请求成功
							resultObj = rpcInput.getRetObj();
						}
					} else {// 有异常时抛出异常
						throw rpcInput.getError().getException();
					}
				} else {
					throw new SOAServerRuntimeException("Token匹配错误，RpcOutput：" + rpcOutput.getName() + ";RpcInput：" + rpcInput.getName());
				}
			}
			t.setStatus(Transaction.SUCCESS);
		} catch (BusinessRuntimeException e) {// 业务异常认为正常请求
			log.warn("Netty框架.SOA客户端.请求告警.Code:" + e.getErrorCode() + "Msg:" + e.getMessage(), e);
			t.setStatus(Transaction.SUCCESS);
			throw e;
		} catch (SOAServerRuntimeException e) {
			log.error("Netty框架.SOA客户端.请求异常.Msg:" + e.getMessage(), e);
			t.setStatus(e);
			throw e;
		} catch (Exception e) {
			log.error("Netty框架.SOA客户端.请求异常.Msg:" + e.getMessage(), e);
			t.setStatus(e);
			throw e;
		} finally {
			t.complete();
		}
		return resultObj;
	}

	/**
	 * 发送请求 重试
	 * 
	 * @param rpcOutput
	 * @return
	 * @throws Exception
	 */
	private RpcInput sendRequestRetry(RpcOutput rpcOutput) throws Exception {
		RpcInput rpcInput = null;
		int i = 0;
		while (i <= rpcOutput.getRetryCount()) {
			try {
				i++;
				rpcInput = this.sendRequest(rpcOutput);
				break;
			} catch (Exception e) {
				log.error("Netty框架.SOA.发送请求重试[InterfaceProxyBase.sendRequestRetry].异常：i=" + i, e);
				if (i > rpcOutput.getRetryCount())
					throw e;
			}
		}
		return rpcInput;
	}

	/**
	 * 发送请求
	 * 
	 * @param objectBase
	 * @return
	 */
	private RpcInput sendRequest(RpcOutput rpcOutput) throws Exception {
		RpcInput rpcInput = null;
		if (rpcOutput != null) {
			Socket socket = null;
			try {
				String serviceHost = this.serviceBase.getServiceHost();
				int servicePort = this.serviceBase.getServicePort();
				socket = new Socket(serviceHost, servicePort);
				if (this.serviceBase.getConnectionTimeout() > 0)// 设置超时时间
					socket.setSoTimeout(this.serviceBase.getConnectionTimeout());
				MessageTree tree = Cat.getManager().getThreadLocalMessageTree();
				// CAT设置Cross信息
				Cat.logEvent("PigeonCall.server", HttpUtil.getIP((InetSocketAddress) socket.getRemoteSocketAddress()));
				Cat.logEvent("PigeonCall.app", this.serviceBase.getAppName());
				Cat.logEvent("PigeonCall.port", String.valueOf(servicePort));
				// 设置LogView消息Id
				String childId = Cat.createMessageId();
				Cat.logEvent(CatConstants.TYPE_REMOTE_CALL, "", Event.SUCCESS, childId);
				rpcOutput.setRootMessageId(tree.getRootMessageId());
				rpcOutput.setMessageId(tree.getMessageId());
				rpcOutput.setChildMessageId(childId);

				OutputStream out = socket.getOutputStream();
				InputStream in = socket.getInputStream();
				String jsonObject = gson.toJson(rpcOutput);
				out.write((jsonObject + "\r\n").getBytes(charset_utf8));
				// 当为非异步方法时，才需要等待接收返回值
				if (rpcOutput.getIsAsynMethod().equals(Boolean.toString(false))) {
					byte[] head = new byte[10];
					in.read(head);
					String strLength = new String(head);
					int length = Integer.valueOf(strLength);
					byte[] data = new byte[length];
					int readLength = in.read(data);
					while (readLength < length) {// 当一次读不完时，需分多次读取
						byte[] dataFollow = new byte[length - readLength];
						int dataFollowLength = in.read(dataFollow);
						for (int i = 0; i < dataFollowLength; i++) {
							byte b = dataFollow[i];
							data[readLength + i] = b;
						}
						readLength += dataFollowLength;
						log.debug("Netty框架.SOA.发送请求[InterfaceProxyBase.sendRequest],length:" + length + ",readLength:" + readLength + ",dataFollowLength:"
								+ dataFollowLength + ";dataFollowStr:" + new String(dataFollow, charset_utf8));
					}
					String strData = new String(data, charset_utf8);
					log.debug("Netty框架.SOA.发送请求[InterfaceProxyBase.sendRequest],length:" + strLength + ";Data:" + strData);
					if (StringUtils.isNotBlank(strData)) {
						rpcInput = gson.fromJson(strData, RpcInput.class);
					}
				} else {
					log.debug("Netty框架.SOA.发送请求[InterfaceProxyBase.sendRequest],该方法为异步方法，无需等待接收返回值");
				}
			} catch (Exception e) {
				throw e;
			} finally {
				if (socket != null) {
					socket.close();
				}
			}
		}
		return rpcInput;
	}

	/**
	 * @return 服务接口
	 */
	public Class getServiceInterface() {
		return serviceInterface;
	}
}
