package com.xunlei.netty.httpserver.cmd.common;

import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.lang.management.RuntimeMXBean;
import java.lang.management.ThreadMXBean;
import java.sql.SQLException;
import java.util.List;

import org.jboss.netty.handler.execution.OrderedMemoryAwareThreadPoolExecutor;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import redis.clients.jedis.JedisPool;

import com.mchange.v2.c3p0.ComboPooledDataSource;
import com.xunlei.netty.cache.JRedisProxy;
import com.xunlei.netty.httpserver.cmd.BaseStatCmd;
import com.xunlei.netty.httpserver.cmd.annotation.CmdAdmin;
import com.xunlei.netty.httpserver.component.XLHttpRequest;
import com.xunlei.netty.httpserver.component.XLHttpResponse;
import com.xunlei.netty.util.HumanReadableUtil;
import com.xunlei.netty.util.Log;
import com.xunlei.netty.util.NettyServerConfig;
import com.xunlei.netty.util.SystemInfo;

/**
 * <pre>
 *  请求次数: 55
 *  响应次数: 53
 *  通道提前关闭: 5
 *  请求次数 - 响应次数 = 通道提前关闭 - ClosedChannelError(提前检查到通道关闭的次数)
 * 
 *  通道打开次数: 77
 *  通道关闭次数: 76
 *  当前打开通道: 1
 *  服务器清除通道: 0
 *  当前打开通道 = 通道打开次数 - 通道关闭次数
 *  客户端或服务器端 正常关闭次数 = 通道关闭次数 - 服务器清除通道
 *  
 *  401IP受限: 0           ip过滤不通过的次数,ipfilter
 *  403禁止访问: 26        favicon.ico请求的次数
 *  404未找到: 0           找不到cmd类或方法的次数
 *  408处理超时: 0         内部future.get(toleranceTimeout)超时的次数
 *  500内部错误: 0         业务处理错误次数
 * </pre>
 * 
 * <pre>
 * 开始解码     attach.decode           (req.createTime)        
 *                                  } 解码用时
 * 开始业务处理   attach.process          (resp.createTime)
 *                                  } 业务处理用时
 * 开始编码     attach.encode
 *                                  } 编码用时
 * 发送完成     attach.complete
 * </pre>
 * 
 * @author ZengDong
 * @since 2010-5-20 下午03:34:28
 */
@Service
public class StatCmd extends BaseStatCmd {
	private static final Logger log = Log.getLogger();
	@Autowired
	private NettyServerConfig config;

	@CmdAdmin
	public Object process(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		StringBuilder tmp = new StringBuilder();
		tmp.append(this.getSytemInfo());
		return tmp.toString();
	}

	private String getSytemInfo() {
		StringBuilder tmp = new StringBuilder();
		RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
		tmp.append("runtimeMXBean\n");
		tmp.append("启动时间:\t\t").append(SystemInfo.getServerStartupTime()).append("\n");
		tmp.append("已运行:\t\t\t").append(HumanReadableUtil.timeSpan(runtimeMXBean.getUptime())).append("\n");
		tmp.append("进程:\t\t\t").append(runtimeMXBean.getName()).append("\n");
		tmp.append("名称:\t\t\t").append(SystemInfo.LAUNCHER_NAME).append("\n");
		tmp.append("虚拟机:\t\t\t").append(runtimeMXBean.getVmName()).append(" ").append(runtimeMXBean.getVmVersion()).append("\n");

		ThreadMXBean threadMXBean = ManagementFactory.getThreadMXBean();
		tmp.append("\n");
		tmp.append("ThreadMXBean\n");
		tmp.append("当前线程数:\t\t").append(threadMXBean.getThreadCount()).append("\n");
		tmp.append("后台线程数:\t\t").append(threadMXBean.getDaemonThreadCount()).append("\n");
		tmp.append("峰值线程数:\t\t").append(threadMXBean.getPeakThreadCount()).append("\n");
		tmp.append("已启动线程数:\t\t").append(threadMXBean.getTotalStartedThreadCount()).append("\n");
		tmp.append("\n");

		OrderedMemoryAwareThreadPoolExecutor pipelineExecutor = config.getPipelineExecutor();
		tmp.append("\n");
		tmp.append("Netty业务线程池[PipelineExecutor]\n");
		tmp.append("ActiveCount:\t\t").append(pipelineExecutor.getActiveCount()).append("\n");
		tmp.append("CompletedTaskCount:\t").append(pipelineExecutor.getCompletedTaskCount()).append("\n");
		tmp.append("CorePoolSize:\t\t").append(pipelineExecutor.getCorePoolSize()).append("\n");
		tmp.append("LargestPoolSize:\t").append(pipelineExecutor.getLargestPoolSize()).append("\n");
		tmp.append("MaximumPoolSize:\t").append(pipelineExecutor.getMaximumPoolSize()).append("\n");
		tmp.append("PoolSize:\t\t").append(pipelineExecutor.getPoolSize()).append("\n");
		tmp.append("MaxChannelMemorySize:\t").append(pipelineExecutor.getMaxChannelMemorySize()).append("\n");
		tmp.append("MaxTotalMemorySize:\t").append(pipelineExecutor.getMaxTotalMemorySize()).append("\n");
		int queueCount = 0;
		if (pipelineExecutor.getQueue() != null && pipelineExecutor.getQueue().size() > 0)
			queueCount = pipelineExecutor.getQueue().size();
		tmp.append("QueueCount:\t\t").append(queueCount).append("\n");
		tmp.append("TaskCount:\t\t").append(pipelineExecutor.getTaskCount()).append("\n");
		tmp.append("\n");

		if (JRedisProxy.getInstance() != null) {
			JedisPool jedisPool = JRedisProxy.getInstance().getJedisPool();
			if (jedisPool != null) {
				tmp.append("\n");
				tmp.append("Redis线程池[JedisPool]\n");
				tmp.append("最大连接数[JRedisProxy.MaxTotal]:\t\t").append(JRedisProxy.getInstance().getMaxTotal()).append("\n");
				tmp.append("最大空闲连接数[JRedisProxy.MaxIdle]:\t\t").append(JRedisProxy.getInstance().getMaxIdle()).append("\n");
				tmp.append("最大等待毫秒数[JRedisProxy.MaxWaitMillis]:\t").append(JRedisProxy.getInstance().getMaxWaitMillis()).append("\n");
				tmp.append("NumActive:\t\t\t\t\t").append(jedisPool.getNumActive()).append("\n");
				tmp.append("NumIdle:\t\t\t\t\t").append(jedisPool.getNumIdle()).append("\n");
				tmp.append("NumWaiters:\t\t\t\t\t").append(jedisPool.getNumWaiters()).append("\n");
				tmp.append("MaxBorrowWaitTimeMillis:\t\t\t").append(jedisPool.getMaxBorrowWaitTimeMillis()).append("\n");
				tmp.append("MeanBorrowWaitTimeMillis:\t\t\t").append(jedisPool.getMeanBorrowWaitTimeMillis()).append("\n");
				tmp.append("\n");
			}
		}
		
		//C3p0线程池
		if(config.getC3p0ComboPool() != null && config.getC3p0ComboPool().size() > 0){
			tmp.append("\n");
			for (ComboPooledDataSource c3p0Pool : config.getC3p0ComboPool()) {
				tmp.append("C3p0线程池[C3p0ComboPool]---").append(c3p0Pool.getJdbcUrl()).append("\n");
				tmp.append("最大连接数[C3p0ComboPool.MaxPoolSize]:\t\t\t\t").append(c3p0Pool.getMaxPoolSize()).append("\n");
				tmp.append("最小连接数[C3p0ComboPool.MinPoolSize]:\t\t\t\t").append(c3p0Pool.getMinPoolSize()).append("\n");
				tmp.append("初始化时连接数[C3p0ComboPool.InitialPoolSize]:\t\t\t").append(c3p0Pool.getInitialPoolSize()).append("\n");
				tmp.append("等待获取新连接时间(ms)[C3p0ComboPool.CheckoutTimeout]:\t\t").append(c3p0Pool.getCheckoutTimeout()).append("\n");
				tmp.append("最大空闲时间(s)[C3p0ComboPool.MaxIdleTime]:\t\t\t").append(c3p0Pool.getMaxIdleTime()).append("\n");
				tmp.append("同时获取的连接数[C3p0ComboPool.AcquireIncrement]:\t\t").append(c3p0Pool.getAcquireIncrement()).append("\n");
				try {
					tmp.append("总连接数[C3p0ComboPool.NumConnections]:\t\t\t\t").append(c3p0Pool.getNumConnections()).append("\n");
					tmp.append("忙连接数[C3p0ComboPool.NumBusyConnections]:\t\t\t").append(c3p0Pool.getNumBusyConnections()).append("\n");
					tmp.append("空闲连接数[C3p0ComboPool.NumIdleConnections]:\t\t\t").append(c3p0Pool.getNumIdleConnections()).append("\n");
				} catch (SQLException e) {
					log.error("获取C3p0连接池[StatCmd.getSytemInfo]异常",e);
				}
				tmp.append("\n");
			}
			tmp.append("\n");
		}

		SystemInfo.getMemoryInfo(tmp);
		SystemInfo.getGarbageCollectorInfo(tmp);

		OperatingSystemMXBean operatingSystemMXBean = ManagementFactory.getOperatingSystemMXBean();
		tmp.append("\n");
		tmp.append("operatingSystemMXBean\n");
		tmp.append("操作系统:\t\t").append(operatingSystemMXBean.getName()).append(" ").append(operatingSystemMXBean.getVersion()).append("\n");
		tmp.append("体系结构:\t\t").append(operatingSystemMXBean.getArch()).append("\n");
		tmp.append("处理器个数:\t\t").append(operatingSystemMXBean.getAvailableProcessors()).append("\n");
		return tmp.toString();
	}
}
