/**
 * 
 */
package com.xunlei.netty.grpcserver.nameresolver;

import java.net.InetSocketAddress;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Vector;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;

import com.dianping.cat.Cat;
import com.dianping.cat.message.Transaction;
import com.ecwid.consul.v1.health.model.HealthService;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.xunlei.netty.consul.ConsulFactory;
import com.xunlei.netty.consul.service.ServiceClient;
import com.xunlei.netty.soaserver.component.SOAServiceBase;
import com.xunlei.netty.util.GRPCServerHelper;
import com.xunlei.netty.util.Log;

import io.grpc.Attributes;
import io.grpc.NameResolver;
import io.grpc.ResolvedServerInfo;
import io.grpc.Status;
import io.grpc.internal.LogExceptionRunnable;
import io.grpc.internal.SharedResourceHolder;
import io.grpc.internal.SharedResourceHolder.Resource;

/**
 * Consul 服务名 解析器
 * 
 * @author wangcanyi
 *
 */
public class ConsulNameResolver extends NameResolver {
	private static final Logger log = Log.getLogger();
	private final String authority;
	private final String serviceName;
	private final Resource<ScheduledExecutorService> timerServiceResource;
	private final Resource<ExecutorService> executorResource;
	@GuardedBy("this")
	private boolean shutdown;
	@GuardedBy("this")
	private ScheduledExecutorService timerService;
	@GuardedBy("this")
	private ExecutorService executor;
	@GuardedBy("this")
	private ScheduledFuture<?> resolutionTask;
	@GuardedBy("this")
	private boolean resolving;
	@GuardedBy("this")
	private Listener listener;
	private List<InetSocketAddress> availableServiceList = new ArrayList<InetSocketAddress>();
	private final Object lock = new Object();
	private SOAServiceBase soaServiceBase;

	ConsulNameResolver(@Nullable String nsAuthority, String name, Attributes params, Resource<ScheduledExecutorService> timerServiceResource,
			Resource<ExecutorService> executorResource, SOAServiceBase soaServiceBase) {
		this.timerServiceResource = timerServiceResource;
		this.executorResource = executorResource;
		URI nameUri = URI.create("//" + name);
		authority = Preconditions.checkNotNull(nameUri.getAuthority(), "nameUri (%s) doesn't have an authority", nameUri);
		serviceName = name;
		this.soaServiceBase = soaServiceBase;
	}

	@Override
	public final String getServiceAuthority() {
		return authority;
	}

	@Override
	public final synchronized void start(Listener listener) {
		Preconditions.checkState(this.listener == null, "already started");
		timerService = SharedResourceHolder.get(timerServiceResource);
		executor = SharedResourceHolder.get(executorResource);
		this.listener = Preconditions.checkNotNull(listener, "listener");
		resolve();
		if (this.soaServiceBase.isConsulIsEnable()) {// 当启用Consul时才需要定时更新
			resolutionTask = timerService.scheduleAtFixedRate(new LogExceptionRunnable(resolutionRunnableOnExecutor), ServiceClient.getCheckInterval(),
					ServiceClient.getCheckInterval(), TimeUnit.SECONDS);
		}
	}

	@Override
	public final synchronized void refresh() {
		Preconditions.checkState(listener != null, "not started");
		resolve();
	}

	private final Runnable resolutionRunnable = new Runnable() {
		@Override
		public void run() {
			Transaction t = Cat.newTransaction("GRPC", "ConsulNameResolver.resolutionRunnable");
			t.addData("serviceName", serviceName);
			Listener savedListener;
			synchronized (ConsulNameResolver.this) {
				if (shutdown) {
					return;
				}
				savedListener = listener;
				resolving = true;
			}
			try {
				if (this.isServiceListUpdate()) {
					if (availableServiceList != null && availableServiceList.size() > 0) {
						List<InetSocketAddress> asList = new ArrayList<InetSocketAddress>(availableServiceList.size());
						asList.addAll(availableServiceList);
						List<List<ResolvedServerInfo>> serverList = new ArrayList<List<ResolvedServerInfo>>(availableServiceList.size());
						List<ResolvedServerInfo> servers = null;
						for (InetSocketAddress inetSocketAddress : asList) {
							servers = new ArrayList<ResolvedServerInfo>(1);
							servers.add(new ResolvedServerInfo(inetSocketAddress, Attributes.EMPTY));
							serverList.add(servers);
						}
						savedListener.onUpdate(serverList, Attributes.EMPTY);
					} else {
						savedListener.onError(Status.UNAVAILABLE.withCause(new Exception("服务可用列表为空")));
					}
				}
				t.setStatus(Transaction.SUCCESS);
			} catch (Exception e) {
				t.setStatus(e);
				log.error("GRPC.ConsulNameResolver.resolutionRunnable.异常", e);
			} finally {
				t.complete();
				synchronized (ConsulNameResolver.this) {
					resolving = false;
				}
			}
		}

		/**
		 * 服务列表是否更改
		 * 
		 * @return
		 */
		private boolean isServiceListUpdate() {
			List<InetSocketAddress> consullist = null;
			Map<String,Integer> healthServiceList = null;
			if (soaServiceBase.isConsulIsEnable()) {// 如果启用Consul时，获取健康服务
				healthServiceList = getServiceList(serviceName);
			} else {// 不启用Consul时，使用自定义地址
				if (StringUtils.isNotBlank(soaServiceBase.getGrpcServiceHost()) && soaServiceBase.getGrpcServicePort() > 0) {
					healthServiceList = new HashMap<String, Integer>(1);
					healthServiceList.put(soaServiceBase.getGrpcServiceHost(),soaServiceBase.getGrpcServicePort());
				}
			}
			if (healthServiceList == null)// 获取健康服务异常时，保持原样不修改原始服务
				return false;
			if (healthServiceList.size() > 0) {
				consullist = new ArrayList<InetSocketAddress>(healthServiceList.size());
				for (Entry<String, Integer> service : healthServiceList.entrySet()) {
					InetSocketAddress socketAddress = new InetSocketAddress(service.getKey(), service.getValue());
					consullist.add(socketAddress);
				}
			}
			if (!GRPCServerHelper.equalsList(consullist, availableServiceList)) {// 如果不相等需要修改,则更新availableServiceList
				synchronized (lock) {
					if (!GRPCServerHelper.equalsList(consullist, availableServiceList)) {
						availableServiceList.clear();
						if (consullist != null && consullist.size() > 0) {
							availableServiceList.addAll(consullist);
						}
						return true;
					} else {
						return false;
					}
				}
			} else {
				return false;
			}
		}
	};

	private final Runnable resolutionRunnableOnExecutor = new Runnable() {
		@Override
		public void run() {
			synchronized (ConsulNameResolver.this) {
				if (!shutdown) {
					executor.execute(resolutionRunnable);
				}
			}
		}
	};

	@VisibleForTesting
	Map<String,Integer> getServiceList(String serviceName) {
		Map<String,Integer> serviceMap = null;
		Transaction t = Cat.newTransaction("GRPC", "ConsulNameResolver.getServiceList");
		t.addData("serviceName", serviceName);
		try {
			if (StringUtils.isNotBlank(serviceName)) {
				List<HealthService> hsList = ConsulFactory.getInstance().getConsulClient().getHealthServices(serviceName, true, null).getValue();
				if (hsList != null && hsList.size() > 0) {
					serviceMap = new HashMap<String, Integer>();
					for (HealthService healthService : hsList) {
						if (!serviceMap.containsKey(healthService.getService().getAddress()))
							serviceMap.put(healthService.getService().getAddress(), healthService.getService().getPort());
					}
				}
			}
			t.setStatus(Transaction.SUCCESS);
		} catch (Exception e) {
			t.setStatus(e);
			log.error("GRPC.ConsulNameResolver.getServiceList.异常：" + e.getMessage(), e);
		} finally {
			t.complete();
		}
		return serviceMap;
	}

	@GuardedBy("this")
	private void resolve() {
		if (resolving || shutdown) {
			return;
		}
		executor.execute(resolutionRunnable);
	}

	@Override
	public final synchronized void shutdown() {
		if (shutdown) {
			return;
		}
		shutdown = true;
		if (resolutionTask != null) {
			resolutionTask.cancel(false);
		}
		if (timerService != null) {
			timerService = SharedResourceHolder.release(timerServiceResource, timerService);
		}
		if (executor != null) {
			executor = SharedResourceHolder.release(executorResource, executor);
		}
	}
}
