/**
 * 
 */
package com.xunlei.netty.grpcserver.interceptor;

import org.slf4j.Logger;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.spi.MessageTree;
import com.xunlei.netty.exception.BusinessRuntimeException;
import com.xunlei.netty.util.GRPCServerHelper;
import com.xunlei.netty.util.Log;
import com.xunlei.netty.util.NettyServerConfig;
import com.xunlei.netty.util.spring.BeanUtil;
import com.xunlei.niux.common.util.IPUtil;
import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCall.Listener;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.ForwardingServerCall.SimpleForwardingServerCall;
import io.grpc.ForwardingServerCallListener.SimpleForwardingServerCallListener;
import io.grpc.Status;

/**
 * Cat Server 拦截器
 * 
 * @author wangcanyi
 *
 */
public class CatServerInterceptor implements ServerInterceptor {
	private static final Logger log = Log.getLogger();
	private static NettyServerConfig serverConfig = BeanUtil.getTypedBean(NettyServerConfig.class);

	@Override
	public <ReqT, RespT> Listener<ReqT> interceptCall(final ServerCall<ReqT, RespT> call, final Metadata headers, ServerCallHandler<ReqT, RespT> next) {

		return new SimpleForwardingServerCallListener<ReqT>(next.startCall(new SimpleForwardingServerCall<ReqT, RespT>(call) {
			@Override
			public void sendHeaders(Metadata responseHeaders) {
				String serviceIP = IPUtil.getLocalHostAddress();
				String servicePort = "";
				if (serverConfig != null)
					servicePort = serverConfig.getListen_grpc_port() + "";
				responseHeaders.put(GRPCServerHelper.MDKey_ServiceIPKey, serviceIP);
				responseHeaders.put(GRPCServerHelper.MDKey_ServicePortKey, servicePort);
				super.sendHeaders(responseHeaders);
			}
		}, headers)) {
			@Override
			public void onHalfClose() {
				Transaction t = Cat.newTransaction("PigeonService", call.getMethodDescriptor().getFullMethodName());
				String clientIP = "";
				if (headers.containsKey(GRPCServerHelper.MDKey_ClientIP)) {
					clientIP = headers.get(GRPCServerHelper.MDKey_ClientIP);
				}
				t.addData("ClientIp", clientIP);
				t.addData("ServerIp", IPUtil.getLocalHostAddress());
				Cat.logEvent("PigeonService.client", clientIP);
				// 设置cross信息
				String appName = "";
				if (serverConfig != null)
					appName = headers.get(GRPCServerHelper.MDKey_ClientAppName);
				Cat.logEvent("PigeonService.app", appName);
				// 设置logview信息
				MessageTree tree = Cat.getManager().getThreadLocalMessageTree();
				if (headers.containsKey(GRPCServerHelper.MDKey_ChildId))
					tree.setMessageId(headers.get(GRPCServerHelper.MDKey_ChildId));
				if (headers.containsKey(GRPCServerHelper.MDKey_RootMessageId))
					tree.setRootMessageId(headers.get(GRPCServerHelper.MDKey_RootMessageId));
				if (headers.containsKey(GRPCServerHelper.MDKey_MessageId))
					tree.setParentMessageId(headers.get(GRPCServerHelper.MDKey_MessageId));

				Status status = Status.OK;
				Metadata metadata = new Metadata();
				try {
					this.delegate().onHalfClose();
					if (call.isCancelled())
						throw Status.CANCELLED.asRuntimeException();
					t.setStatus(Transaction.SUCCESS);
				} catch (BusinessRuntimeException e) {
					t.setStatus(Transaction.SUCCESS);
					status = Status.INTERNAL;
					metadata = GRPCServerHelper.getMetadataFromException(e);
					log.warn("GRPCServer.CatServerInterceptor.onHalfClose.告警:" + GRPCServerHelper.getExceptionStringFromMetadata(metadata), e);
				} catch (Exception e) {
					// 统一处理异常
					t.setStatus(e);
					status = Status.fromThrowable(e);
					Metadata trailers = Status.trailersFromThrowable(e);
					if (trailers == null) {
						// 如果没有找到，即异常中没有设置 Metadata，则只能 new 一个空的 Metadata
						metadata = GRPCServerHelper.getMetadataFromException(e);
					} else {
						metadata = trailers;
					}
					log.error("GRPCServer.CatServerInterceptor.onHalfClose.异常:" + GRPCServerHelper.getExceptionStringFromMetadata(metadata), e);
				} finally {
					t.complete();
					status.withDescription(metadata.toString());
					try {
						call.close(status, metadata);
					} catch (Exception e2) {
						log.error("GRPCServer.CatServerInterceptor.onHalfClose.close异常.MethodName:" + call.getMethodDescriptor().getFullMethodName() + ";", e2);
					}
				}
			}
			
			@Override
			public void onComplete() {
				Transaction t = Cat.newTransaction("GRPC","GRPCServer.CatServerInterceptor.onComplete");
				t.addData("FullMethodName",call.getMethodDescriptor().getFullMethodName());
				super.onComplete();
				t.setStatus(Transaction.SUCCESS);
				t.complete();
			}
			
			@Override
			public void onMessage(ReqT message) {
				Transaction t = Cat.newTransaction("GRPC","GRPCServer.CatServerInterceptor.onMessage");
				t.addData("FullMethodName",call.getMethodDescriptor().getFullMethodName());
				super.onMessage(message);
				t.setStatus(Transaction.SUCCESS);
				t.complete();
			}
			
			@Override
			public void onReady() {
				Transaction t = Cat.newTransaction("GRPC","GRPCServer.CatServerInterceptor.onReady");
				t.addData("FullMethodName",call.getMethodDescriptor().getFullMethodName());
				super.onReady();
				t.setStatus(Transaction.SUCCESS);
				t.complete();
			}

			@Override
			public void onCancel() {// 调用被取消时
				Transaction t = Cat.newTransaction("GRPC","GRPCServer.CatServerInterceptor.onCancel");
				t.addData("FullMethodName",call.getMethodDescriptor().getFullMethodName());
				super.onCancel();
				t.setStatus(Transaction.SUCCESS);
				t.complete();
			}
		};
	}
}
