/**
 * 
 */
package com.xunlei.netty.grpcserver.interceptor;

import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;

import com.dianping.cat.Cat;
import com.dianping.cat.CatConstants;
import com.dianping.cat.message.Event;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.spi.MessageTree;
import com.xunlei.netty.grpcserver.nameresolver.ConsulNameResolverProvider;
import com.xunlei.netty.util.GRPCServerHelper;
import com.xunlei.netty.util.Log;
import com.xunlei.niux.common.util.IPUtil;

import io.grpc.CallOptions;
import io.grpc.Channel;
import io.grpc.ClientCall;
import io.grpc.ClientInterceptor;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import io.grpc.ForwardingClientCall.SimpleForwardingClientCall;
import io.grpc.ForwardingClientCallListener.SimpleForwardingClientCallListener;
import io.grpc.Metadata;
import io.grpc.StatusException;

/**
 * CAT Client 拦截器
 * 
 * @author wangcanyi
 *
 */
public class CatClientInterceptor implements ClientInterceptor {
	private static final Logger log = Log.getLogger();

	@Override
	public <ReqT, RespT> ClientCall<ReqT, RespT> interceptCall(MethodDescriptor<ReqT, RespT> method, CallOptions callOptions, Channel next) {
		long connectionTimeout = callOptions.getOption(GRPCServerHelper.COKey_ConnectionTimeout);
		if (connectionTimeout > 0) {// 接口超时设置
			callOptions = callOptions.withDeadlineAfter(connectionTimeout, TimeUnit.MILLISECONDS);
		}
		String fullMethodName = method.getFullMethodName();
		String authority = next.authority();// 应用名
		final boolean isAsyncInterface = callOptions.getOption(GRPCServerHelper.COKey_IsAsyncInterface);
		final Transaction t = Cat.newTransaction("PigeonCall", fullMethodName);
		Cat.logEvent("PigeonCall.app", authority);
		return new SimpleForwardingClientCall<ReqT, RespT>(next.newCall(method, callOptions)) {
			@Override
			public void start(Listener<RespT> responseListener, Metadata headers) {
				if (headers != null) {
					MessageTree tree = Cat.getManager().getThreadLocalMessageTree();
					// 设置LogView消息Id
					String childId = Cat.createMessageId();
					Cat.logEvent(CatConstants.TYPE_REMOTE_CALL, "", Event.SUCCESS, childId);
					String rootMessageId = tree.getRootMessageId();
					String messageId = tree.getMessageId();
					if (messageId == null) {
						messageId = Cat.createMessageId();
						tree.setMessageId(messageId);
					}
					if (rootMessageId == null) {
						rootMessageId = messageId;
					}
					headers.put(GRPCServerHelper.MDKey_ClientIP, IPUtil.getLocalHostAddress() + "");
					headers.put(GRPCServerHelper.MDKey_ClientAppName, Cat.getManager().getDomain());
					headers.put(GRPCServerHelper.MDKey_ChildId, childId);
					if (rootMessageId != null)
						headers.put(GRPCServerHelper.MDKey_RootMessageId, rootMessageId);
					if (messageId != null)
						headers.put(GRPCServerHelper.MDKey_MessageId, messageId);
				}
				super.start(new SimpleForwardingClientCallListener<RespT>(responseListener) {
					@Override
					public void onHeaders(Metadata headers) {
						String serviceIP = "";
						String servicePort = "";
						if (headers.containsKey(GRPCServerHelper.MDKey_ServiceIPKey))
							serviceIP = headers.get(GRPCServerHelper.MDKey_ServiceIPKey);
						if (headers.containsKey(GRPCServerHelper.MDKey_ServicePortKey))
							servicePort = headers.get(GRPCServerHelper.MDKey_ServicePortKey);
						Cat.logEvent("PigeonCall.server", serviceIP);
						Cat.logEvent("PigeonCall.port", servicePort);
						super.onHeaders(headers);
					}

					@Override
					public void onClose(Status status, Metadata trailers) {
						if (!isAsyncInterface) {// 同步接口时，才进行CAT记录
							if (status.isOk()) {
								t.setStatus(Transaction.SUCCESS);
							} else {
								StatusException se = status.asException(trailers);
								t.setStatus(se);
								log.error("GRPCServer.CatClientInterceptor.onClose.异常:" + GRPCServerHelper.getExceptionStringFromMetadata(trailers), se);
							}
							t.complete();
						}
						super.onClose(status, trailers);
					}
				}, headers);
			}

			@Override
			public void halfClose() {
				super.halfClose();
				if (isAsyncInterface) {// 当为异常方法时，无需等待返回值，则关关闭时，认为已经成功
					t.setStatus(Transaction.SUCCESS);
					t.complete();
				}
			}

			@Override
			public void cancel(String message, Throwable cause) {
				if (!t.isCompleted()){
					t.setStatus(cause);
					t.complete();// 异常时导致关闭
					log.error("GRPCServer.CatClientInterceptor.cancel.异常:" + message, cause);
				}
				super.cancel(message, cause);
			}
		};
	}
}
