/**
 * 
 */
package com.xunlei.netty.grpcserver.client;

import java.lang.reflect.Method;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;

import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.stub.AbstractStub;
import io.grpc.util.RoundRobinLoadBalancerFactory;

import com.xunlei.netty.grpcserver.interceptor.CatClientInterceptor;
import com.xunlei.netty.grpcserver.nameresolver.ConsulNameResolverProvider;
import com.xunlei.netty.soaserver.component.SOAServiceBase;
import com.xunlei.netty.util.GRPCServerHelper;
import com.xunlei.netty.util.Log;

/**
 * GRPC客户端工厂类
 * 
 * @author wangcanyi
 *
 */
public class GRPCClientFactory {
	private static final Logger log = Log.getLogger();
	private SOAServiceBase serviceBase;
	private ManagedChannel grpcChannel;

	/**
	 * 构造器
	 * 
	 * @param serviceBase 服务基类
	 */
	public GRPCClientFactory(SOAServiceBase serviceBase) {
		this.serviceBase = serviceBase;
		this.grpcChannel = ManagedChannelBuilder.forTarget("consul://" + this.serviceBase.getAppName())
				.nameResolverFactory(new ConsulNameResolverProvider(serviceBase)).usePlaintext(true)
				.loadBalancerFactory(RoundRobinLoadBalancerFactory.getInstance()).build();

	}

	/**
	 * 创建同步接口
	 * 
	 * @param grpcServiceInterface
	 * @return
	 * @throws Exception
	 */
	public AbstractStub<?> newBlockingStub(Class grpcServiceInterface) {
		AbstractStub<?> stbu = null;
		if (grpcServiceInterface == null)
			throw new RuntimeException("grpcServiceInterface为空");
		try {
			Method method = grpcServiceInterface.getMethod("newBlockingStub", io.grpc.Channel.class);
			if (method != null) {
				stbu = (AbstractStub<?>) method.invoke(null, this.grpcChannel);
				stbu = this.stubWithCommon(stbu).withOption(GRPCServerHelper.COKey_IsAsyncInterface, false);
			}
		} catch (Exception e) {
			e.printStackTrace();
			log.error("GRPC客户端工厂类实例GRPC接口[newBlockingStub].异常.接口名：" + grpcServiceInterface.getName(), e);
		}
		return stbu;
	}

	/**
	 * 创建异步接口
	 * 
	 * @param grpcServiceInterface
	 * @return
	 * @throws Exception
	 */
	public AbstractStub<?> newFutureStub(Class grpcServiceInterface) {
		AbstractStub<?> stbu = null;
		if (grpcServiceInterface == null)
			throw new RuntimeException("grpcServiceInterface为空");
		try {
			Method method = grpcServiceInterface.getMethod("newFutureStub", io.grpc.Channel.class);
			if (method != null) {
				stbu = (AbstractStub<?>) method.invoke(null, this.grpcChannel);
				stbu = this.stubWithCommon(stbu).withOption(GRPCServerHelper.COKey_IsAsyncInterface, true);
			}
		} catch (Exception e) {
			e.printStackTrace();
			log.error("GRPC客户端工厂类实例GRPC接口[newFutureStub].异常.接口名：" + grpcServiceInterface.getName(), e);
		}
		return stbu;
	}

	/**
	 * 创建接口公共信息
	 * 
	 * @param stub
	 * @return
	 */
	private AbstractStub<?> stubWithCommon(AbstractStub<?> stub) {
		stub = stub.withInterceptors(new CatClientInterceptor());
		long connectionTimeout = this.serviceBase.getConnectionTimeout();
		if (connectionTimeout > 0) {
			stub = stub.withOption(GRPCServerHelper.COKey_ConnectionTimeout, connectionTimeout);
		}
		return stub;
	}

	/**
	 * 关闭管道
	 * 
	 * @throws InterruptedException
	 */
	public void shutdowm() throws InterruptedException {
		if (this.grpcChannel != null)
			this.grpcChannel.shutdown().awaitTermination(5, TimeUnit.SECONDS);
	}
}
