/**
 * 
 */
package com.xunlei.netty.consul.service;

import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;

import com.dianping.cat.Cat;
import com.dianping.cat.message.Transaction;
import com.ecwid.consul.v1.QueryParams;
import com.ecwid.consul.v1.agent.model.NewService;
import com.ecwid.consul.v1.agent.model.NewService.Check;
import com.ecwid.consul.v1.health.model.HealthService;
import com.xunlei.netty.consul.ConsulFactory;
import com.xunlei.netty.consul.ConsulFactory.Env;
import com.xunlei.netty.consul.exception.ConsulRuntimeException;
import com.xunlei.netty.util.Log;
import com.xunlei.niux.common.util.IPUtil;

/**
 * 服务注册客户端
 * 
 * @author wangcanyi
 *
 */
public class ServiceClient {
	private static final Logger log = Log.getLogger();

	/**
	 * 服务列表集合,通过定时任务来进行更新
	 */
	private static final ConcurrentHashMap<String, List<HealthService>> mapServiceList = new ConcurrentHashMap<String, List<HealthService>>();

	/**
	 * 健康检测时间间隔，单位秒
	 */
	private static final int CHECK_INTERVAL = 5;
	/**
	 * 健康检测超时时间，单位秒
	 */
	private static final int CHECK_TIMEOUT = 2;

	/**
	 * 注册服务
	 * 
	 * @param serviceName
	 * @param httpPort
	 * @return
	 * @throws Exception 
	 */
	public NewService registerService(String serviceName, int httpPort, int soaPort) throws Exception {
		NewService service = null;
		Transaction t = Cat.newTransaction("Consul", "ServiceClient.registerService");
		try {
			if (StringUtils.isBlank(serviceName))
				throw new ConsulRuntimeException("服务名为空");
			if (httpPort <= 0)
				throw new ConsulRuntimeException("HTTP端口号小于等于0");
			String hostAddress = IPUtil.getLocalHostAddress();// 获取本机IP地址
			Env env = ConsulFactory.CONSUL_ENV;
			if (env == Env.PreRelease || env == Env.Release) {// 正式环境/预发布环境，获取机器名
				hostAddress = IPUtil.getLocalHostName() + ".sandai.net";
			}
			Cat.logEvent("Consul.RegisterService.serviceName", serviceName);
			Cat.logEvent("Consul.RegisterService.hostAddress", hostAddress);
			Cat.logEvent("Consul.RegisterService.httpPort", httpPort + "");
			Cat.logEvent("Consul.RegisterService.soaPort", soaPort + "");
			service = new NewService();
			service.setId(serviceName + "-" + hostAddress);
			service.setName(serviceName);
			service.setAddress(hostAddress);
			service.setPort(soaPort);
			Check check = new Check();
			check.setHttp(String.format("http://%s:%d/echo", hostAddress, httpPort));
			check.setInterval(CHECK_INTERVAL + "s");
			check.setTimeout(CHECK_TIMEOUT + "s");
			service.setCheck(check);
			ConsulFactory.getInstance().getConsulClient().agentServiceRegister(service);
			t.setStatus(Transaction.SUCCESS);
		} catch (Exception e) {
			t.setStatus(e);
			log.error("Consul 注册服务[ServiceClient.registerService] 异常：" + e.getMessage(), e);
			throw e;
		} finally {
			t.complete();
		}
		return service;
	}

	/**
	 * 获取健康检测通过的服务列表
	 * 
	 * @param serviceName 服务名
	 * @return
	 */
	public List<HealthService> getServiceList(String serviceName) {
		List<HealthService> csList = null;
		Transaction t = Cat.newTransaction("Consul", "ServiceClient.getServiceList");
		t.addData("serviceName", serviceName);
		try {
			if (StringUtils.isNotBlank(serviceName)) {
				if(mapServiceList.containsKey(serviceName)){//存在Key则直接返回
					csList = mapServiceList.get(serviceName);
				}else{
					csList = ConsulFactory.getInstance().getConsulClient().getHealthServices(serviceName, true, null).getValue();
					mapServiceList.put(serviceName, csList);
				}
			}
			t.setStatus(Transaction.SUCCESS);
		} catch (Exception e) {
			t.setStatus(e);
			log.error("Consul 获取服务列表[ServiceClient.getServiceList] 异常：" + e.getMessage(), e);
		} finally {
			t.complete();
		}
		return csList;
	}

	/**
	 * @return 服务列表集合,通过定时任务来进行更新
	 */
	public static ConcurrentHashMap<String, List<HealthService>> getMapServiceList() {
		return mapServiceList;
	}

	/**
	 * @return 健康检测时间间隔，单位秒
	 */
	public static int getCheckInterval() {
		return CHECK_INTERVAL;
	}

	/**
	 * @return 健康检测超时时间，单位秒
	 */
	public static int getCheckTimeout() {
		return CHECK_TIMEOUT;
	}
}
