/**
 * 
 */
package com.xunlei.netty.consul;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;

import com.xunyi.ApplicationHelper;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;

import com.ecwid.consul.v1.ConsulClient;
import com.ecwid.consul.v1.ConsulRawClient;
import com.ecwid.consul.v1.agent.model.NewService;
import com.ecwid.consul.v1.catalog.model.CatalogService;
import com.ecwid.consul.v1.health.model.HealthService;
import com.xunlei.netty.consul.loadbalance.ILoadBalance;
import com.xunlei.netty.consul.loadbalance.RoundRobinLoadBalance;
import com.xunlei.netty.consul.service.ServiceClient;
import com.xunlei.netty.util.Log;
import com.xunlei.niux.common.util.IPUtil;

/**
 * Consul 工厂
 * 
 * @author wangcanyi
 *
 */
public class ConsulFactory {
	private static final Logger log = Log.getLogger();

	/**
	 * Consul 当前环境
	 */
	public static Env CONSUL_ENV = getEnv();
	private static ConsulFactory consulFactory;// 正式环境
	private static ConsulFactory consulFactoryTest;// 测试环境
	private static ConsulFactory consulFactoryPreRelease;// 预发布环境
	private String agentHost;
	private int agentPort;
//	private Properties config = new Properties();
	private ConsulClient consulClient;
	private ServiceClient serviceClient;
	private ILoadBalance loadBalance;

	/**
	 * 获取实例
	 * 
	 * @return
	 */
	public static ConsulFactory getInstance() {
		return getInstance(CONSUL_ENV);
	}

	/**
	 * 获取实例
	 * 
	 * @param env 环境变量
	 * @return
	 */
	public static ConsulFactory getInstance(Env env) {
		if (env == Env.Test) {// 测试环境
			if (consulFactoryTest == null)
				consulFactoryTest = new ConsulFactory(env);
			return consulFactoryTest;
		}
		if (env == Env.PreRelease) {// 预发布环境
			if (consulFactoryPreRelease == null)
				consulFactoryPreRelease = new ConsulFactory(env);
			return consulFactoryPreRelease;
		}
		if (env == Env.Release) {// 正式环境
			if (consulFactory == null)
				consulFactory = new ConsulFactory(env);
			return consulFactory;
		}
		return null;
	}

	/**
	 * ProxyTest测试环境文件存放地址
	 * 
	 * @return
	 */
	protected static String getProxyTestUrl() {
		return "/proxyTesting";
	}

	/**
	 * 预发布环境机器名
	 * 
	 * @return
	 */
	protected static String getPreReleaseHostName() {
		return "twin14543vm10,twin14543vm11";
	}

	/**
	 * 获取当前环境变量
	 * 
	 * @return
	 */
	private static Env getEnv() {
		InputStream is = ConsulFactory.class.getResourceAsStream(getProxyTestUrl());
		if (is != null) {
			String info = "ConsulFactory Default Mode: TESTING";
			log.error(info);
			System.err.println(info);
			return Env.Test;
		}
		String localHostName = IPUtil.getLocalHostName();
		if (StringUtils.isNotBlank(localHostName)) {
			if (getPreReleaseHostName().contains(localHostName)) {
				String info = "ConsulFactory Default Mode: Pre-RELEASE\nIf you want to use testing mode:put an empty file named 'proxyTesting' in your classpath";
				log.error(info);
				System.err.println(info);
				return Env.PreRelease;
			} else {
				String info = "ConsulFactory Default Mode: RELEASE\nIf you want to use testing mode:put an empty file named 'proxyTesting' in your classpath";
				log.error(info);
				System.err.println(info);
				return Env.Release;
			}
		}
		return Env.Test;
	}

	/**
	 * 构造器
	 * 
	 * @param env 环境变量
	 * @throws IOException
	 */
	private ConsulFactory(Env env) {
		try {
//			String consul_config = "";
//			if (env == Env.Test)
//				consul_config = "com/xunlei/netty/consul/conf/consul_test.properties";
//			if (env == Env.PreRelease)
//				consul_config = "com/xunlei/netty/consul/conf/consul_pre_release.properties";
//			if (env == Env.Release)
//				consul_config = "com/xunlei/netty/consul/conf/consul_release.properties";
//			config.load(ConsulFactory.class.getClassLoader().getResourceAsStream(consul_config));

//			this.agentHost = config.getProperty("consul.agentHost", "");
//			this.agentPort = Integer.parseInt(config.getProperty("consul.agentPort", "0"));

			this.agentHost = ApplicationHelper.environment.getProperty("consul.agentHost", "consul.niu.xunlei.com");
			this.agentPort = ApplicationHelper.environment.getProperty("consul.agentPort", Integer.class, 8500);

			// 创建ConsulClient
			ConsulRawClient consulRawClient = new ConsulRawClient(this.agentHost, this.agentPort);
			consulClient = new ConsulClient(consulRawClient);
		} catch (Exception e) {
			System.out.println("ConsulFactory 创建失败");
			e.printStackTrace();
			log.error("ConsulFactory创建[ConsulFactory(Env env)].异常", e);
		}
	}

	/**
	 * 环境变量
	 * 
	 * @author wangcanyi
	 *
	 */
	public enum Env {
		/**
		 * 测试环境
		 */
		Test,
		/**
		 * 正式环境
		 */
		Release,
		/**
		 * 预发布环境
		 */
		PreRelease
	}

	/**
	 * @return Consul 服务地址
	 */
	public String getAgentHost() {
		return agentHost;
	}

	/**
	 * @return Consul 服务端口
	 */
	public int getAgentPort() {
		return agentPort;
	}

	/**
	 * @return Consul 客户端
	 */
	public ConsulClient getConsulClient() {
		return consulClient;
	}

	/**
	 * @return 注册客户端
	 */
	public ServiceClient getServiceClient() {
		if (serviceClient == null) {
			serviceClient = new ServiceClient();
		}
		return serviceClient;
	}

	/**
	 * @return 负载均衡
	 */
	public ILoadBalance getLoadBalance() {
		if (loadBalance == null) {
			loadBalance = new RoundRobinLoadBalance();
		}
		return loadBalance;
	}

	/**
	 * 获取健康检测通过的服务列表
	 * 
	 * @param serviceName 服务名
	 * @return
	 */
	public List<HealthService> getServiceList(String serviceName) {
		return getServiceClient().getServiceList(serviceName);
	}

	/**
	 * 注册服务
	 * 
	 * @param serviceName 服务名
	 * @param httpPort HTTP端口
	 * @param soaPort SOA端口
	 * @return
	 * @throws Exception 
	 */
	public NewService registerService(String serviceName, int httpPort, int soaPort) throws Exception {
		return getServiceClient().registerService(serviceName, httpPort, soaPort);
	}

	/**
	 * 选择一个服务
	 * 
	 * @param serviceName 服务名
	 * @return
	 */
	public HealthService choose(String serviceName) {
		return getLoadBalance().choose(serviceName);
	}
}
