package com.xunlei.netty.cache;

import java.io.InputStream;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.xunlei.netty.util.NettyServerConfig;
import com.xunlei.netty.util.spring.AfterConfig;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

/**
 * JRedis缓存
 * 
 * @author wangcanyi
 *
 */
@Component
public class JRedisProxy {
	
	/**
	 * JRedis属性文件地址
	 */
	private static String jredisPropertiesUrl = "jredis.properties";
	private static JRedisProxy jRedisProxy = null;
	private static Object lock = new Object();
	private static Logger logger = LoggerFactory.getLogger(JRedisProxy.class.getName());
	@Autowired
	private NettyServerConfig nettyServerConfig;

	/**
	 * 获取实例
	 * 
	 * @return
	 */
	public static JRedisProxy getInstance() {
		if (jRedisProxy == null) {
			synchronized (lock) {
				if (jRedisProxy == null) {
					jRedisProxy = new JRedisProxy(jredisPropertiesUrl);
				}
			}
		}
		return jRedisProxy;
	}

	private Properties config = new Properties();
	private JedisPool pool;
	private String serverHost;
	private int serverPort;
	private int serverTimeout;
	private int maxTotal;
	private int maxIdle;
	private int maxWaitMillis;
	private static final String catMetricsType = "Cache.jredis";
	private String serverPassword;

	/**
	 * 无参数构造
	 */
	public JRedisProxy() {

	}

	/**
	 * 参数构造
	 * 
	 * @param jredisPropertiesUrl 属性文件地址
	 * @throws Exception
	 */
	public JRedisProxy(String jredisPropertiesUrl) {
		if (StringUtils.isNotBlank(jredisPropertiesUrl)) {
			System.out.println("jredis system连接 初始化");
			try {
				InputStream jredisProperties = JRedisProxy.class.getClassLoader().getResourceAsStream(jredisPropertiesUrl);
				if (jredisProperties != null) {
					initJedisPool(jredisProperties);
				}
			} catch (Exception e) {
				System.out.println("jredis system连接初始化失败");
				e.printStackTrace();
				logger.error("JRedisProxy初始化[JRedisProxy(jredisPropertiesUrl)].异常", e);
			}
		}
	}

	/**
	 * 初始化Jedis线程池
	 * 
	 * @param jredisProperties
	 * @throws Exception
	 */
	private void initJedisPool(InputStream jredisProperties) throws Exception {
		config.load(jredisProperties);
		JedisPoolConfig poolConfig = new JedisPoolConfig();
		this.maxTotal = Integer.parseInt(config.getProperty("maxActive", "1000"));
		this.maxIdle = Integer.parseInt(config.getProperty("maxIdle", "30"));
		this.maxWaitMillis = Integer.parseInt(config.getProperty("maxWait", "1000"));
		// jedis版本升级 2.1.0 =》 2.9.0 方法变更setMaxActive =》 setMaxTotal
		// poolConfig.setMaxActive(Integer.parseInt(config.getProperty("maxActive", "1000")));
		poolConfig.setMaxTotal(this.maxTotal);
		poolConfig.setMaxIdle(this.maxIdle);
		// jedis版本升级 2.1.0 =》 2.9.0 方法变更setMaxWait =》 setMaxWaitMillis
		// poolConfig.setMaxWait(Integer.parseInt(config.getProperty("maxWait", "1000")));
		poolConfig.setMaxWaitMillis(this.maxWaitMillis);
		poolConfig.setTestOnBorrow(false);
		poolConfig.setTestOnReturn(false);
		serverHost = config.getProperty("host");
		serverPassword = config.getProperty("password");
		serverPort = Integer.parseInt(config.getProperty("port", "6379"));
		serverTimeout = Integer.parseInt(config.getProperty("timeout", "2000"));// 默认2S超时
		if(!StringUtils.isEmpty(serverPassword)){
			pool = new JedisPool(poolConfig, serverHost, serverPort, serverTimeout, serverPassword);
		}else{
			pool = new JedisPool(poolConfig, serverHost, serverPort, serverTimeout);
		}
		logger.debug(String.format("JRedisProxy.初始化线程池[initJedisPool].正常：Host:%s,Port:%s,TimeOut:%s", serverHost, serverPort, serverTimeout));
	}

	/**
	 * 获取Jedis线程池
	 * 
	 * @return
	 */
	public JedisPool getJedisPool() {
		return this.pool;
	}

	/**
	 * @return 服务地址
	 */
	public String getServerHost() {
		return serverHost;
	}

	/**
	 * @return 服务端口
	 */
	public int getServerPort() {
		return serverPort;
	}

	/**
	 * @return 服务超时时间
	 */
	public int getServerTimeout() {
		return serverTimeout;
	}

	/**
	 * @return 最大连接数
	 */
	public int getMaxTotal() {
		return maxTotal;
	}

	/**
	 * @return 最大空闲连接数
	 */
	public int getMaxIdle() {
		return maxIdle;
	}

	/**
	 * @return 获取连接时的最大等待毫秒数
	 */
	public int getMaxWaitMillis() {
		return maxWaitMillis;
	}

	/**
	 * @return jredis配置文件地址
	 */
	public String getJredisPropertiesUrl() {
		return jredisPropertiesUrl;
	}

	/**
	 * 初始化JRedisProxy
	 */
	@AfterConfig
	private void initJRedisProxy() {
		if (StringUtils.isNotBlank(nettyServerConfig.getJredisPropertiesUrl())) {
			jredisPropertiesUrl = nettyServerConfig.getJredisPropertiesUrl();
		}
		logger.info("JRedisProxy服务启动时初始化，配置文件地址：" + jredisPropertiesUrl);
		jRedisProxy = getInstance();
	}

	/**
	 * CAT Transaction记录
	 * 
	 * @param metricsName
	 * @return
	 */
	private Transaction newTransaction(String metricsName) {
		Transaction t = Cat.newTransaction(catMetricsType, metricsName);
		Cat.logEvent(catMetricsType + ".server", serverHost);
		Cat.logEvent(catMetricsType + ".serverPort", serverPort + "");
		Cat.logEvent(catMetricsType + ".serverTimeout", serverTimeout + "");
		return t;
	}

	/**
	 * 获取缓存key
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @return
	 */
	public String getKey(String prefixKey, String keyId) {
		String key = prefixKey;
		if (StringUtils.isNotBlank(keyId)) {
			if (StringUtils.isNotBlank(prefixKey)) {
				key = prefixKey + keyId;
			} else {
				key = keyId;
			}
		}
		return key;
	}

	/**
	 * 获取缓存Key列表
	 * 
	 * @param prefixKey
	 * @param keyIds
	 * @return
	 */
	public String[] getKey(String prefixKey, String... keyIds) {
		String[] array = null;
		if (keyIds != null && keyIds.length > 0) {
			array = new String[keyIds.length];
			for (int i = 0; i < keyIds.length; i++) {
				String keyId = keyIds[i];
				array[i] = this.getKey(prefixKey, keyId);
			}
		}
		return array;
	}

	/**
	 * 设置缓存
	 * 
	 * @param key 键
	 * @param value 值
	 * @param seconds 过期时间
	 * @return
	 */
	public boolean set(String key, String value, int seconds) {
		return this.set(key, null, value, seconds);
	}

	/**
	 * 设置缓存
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param value 值
	 * @param seconds 过期时间
	 * @return
	 */
	public boolean set(String prefixKey, String keyId, String value, int seconds) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)) {
			String metricsName = "[Method-setex] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, value);
			try {
				jedis = pool.getResource();
				jedis.setex(key, seconds, value);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.setex].正常：key=" + key + ",value=" + value + ",seconds=" + seconds);
				return true;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.setex].异常：key=" + key + ",value=" + value + ",seconds=" + seconds, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return false;
	}
	
	/**
	 * 设置缓存(永不过期)。当且仅当key原本不存在的时候才会设置。
	 * 
	 * @param key 键
	 * @param value 值
	 * 
	 * @return 连接异常返回null，设置成功返回true，设置失败返回false
	 */
	public Boolean setnx(String key, String value) {
		return this.setnx(key, null, value);
	}
	
	/**
	 * 设置缓存(永不过期)。当且仅当key原本不存在的时候才会设置。
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param value 值
	 * 
	 * @return 连接异常返回null，设置成功返回true，设置失败返回false
	 */
	public Boolean setnx(String prefixKey, String keyId, String value) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)) {
			String metricsName = "[Method-setnx] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, value);
			try {
				jedis = pool.getResource();
				Long result = jedis.setnx(key, value);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.setnx].正常：key=" + key + ",value=" + value);
				return result == 1L;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.setnx].异常：key=" + key + ",value=" + value, e);
			} finally {
				if (jedis != null) {
					jedis.close();
				}
				t.complete();
			}
		}
		return null;
	}
	
	/**
	 * 设置缓存(永不过期)，并返回设置之前的value。
	 * 
	 * @param key 键
	 * @param value 值
	 * 
	 * @return 设置之前的value
	 */
	public String getSet(String key, String value) {
		return this.getSet(key, null, value);
	}
	
	/**
	 * 设置缓存(永不过期)，并返回设置之前的value。
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param value 值
	 * 
	 * @return 设置之前的value
	 */
	public String getSet(String prefixKey, String keyId, String value) {
		String result = null;
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)) {
			String metricsName = "[Method-getset] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, value);
			try {
				jedis = pool.getResource();
				result = jedis.getSet(key, value);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.getset].正常：key=" + key + ",newValue=" + value + ",oldValue=" + result);
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.getset].异常：key=" + key + ",newValue=" + value + ",oldValue=" + result, e);
			} finally {
				if (jedis != null) {
					jedis.close();
				}
				t.complete();
			}
		}
		return result;
	}
	
	/**
	 * 设置缓存(<b>要求Redis版本至少是2.6.12</b>)
	 * 
	 * @param key 键
	 * @param value 值
	 * @param nxxx "NX"或"XX"。"NX"：仅当key不存在时才设置value；"XX"：仅当key已存在时才设置value。
	 * @param expx "EX"或"PX"。"EX"：参数time的单位是秒；"PX"：参数time的单位是毫秒。
	 * @param time 过期时间
	 * 
	 * @return 状态码。OK：表示设置成功；(nil)：表示因不满足"NX"或"XX"的条件导致设置失败。
	 */
	public String set(String key, String value, String nxxx, String expx, int time) {
		return this.set(key, null, value, nxxx, expx, time);
	}
	
	/**
	 * 设置缓存(<b>要求Redis版本至少是2.6.12</b>)
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param value 值
	 * @param nxxx "NX"或"XX"。"NX"：仅当key不存在时才设置value；"XX"：仅当key已存在时才设置value。
	 * @param expx "EX"或"PX"。"EX"：参数time的单位是秒；"PX"：参数time的单位是毫秒。
	 * @param time 过期时间
	 * 
	 * @return 状态码。OK：表示设置成功；(nil)：表示因不满足"NX"或"XX"的条件导致设置失败。
	 */
	public String set(String prefixKey, String keyId, String value, String nxxx, String expx, int time) {
		String result = null;
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)) {
			String metricsName = "[Method-set] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, value);
			try {
				jedis = pool.getResource();
				result = jedis.set(key, value, nxxx, expx, time);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.set].正常：key=" + key + ",value=" + value
						+ ",nxxx=" + nxxx + ",expx=" + expx + ",time=" + time + ",result=" + result);
				return result;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.set].异常：key=" + key + ",value=" + value
						+ ",nxxx=" + nxxx + ",expx=" + expx + ",time=" + time + ",result=" + result, e);
			} finally {
				if (jedis != null) {
					jedis.close();
				}
				t.complete();
			}
		}
		return result;
	}

	/**
	 * 设置缓存（永不过期）
	 * 
	 * @param key 键
	 * @param value 值
	 * @return
	 */
	public boolean set(String key, String value) {
		return this.set(key, null, value);
	}

	/**
	 * 设置缓存（永不过期）
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param value 值
	 * @return
	 */
	public boolean set(String prefixKey, String keyId, String value) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)) {
			String metricsName = "[Method-set] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, value);
			try {
				jedis = pool.getResource();
				jedis.set(key, value);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存（永不过期）[JRedisProxy.set].正常：key=" + key + ",value=" + value);
				return true;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存（永不过期）[JRedisProxy.set].异常：key=" + key + ",value=" + value, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return false;
	}

	/**
	 * 获取缓存
	 * 
	 * @param key 键
	 * @return
	 */
	public String get(String key) {
		return this.get(key, null);
	}

	/**
	 * 获取缓存
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @return
	 */
	public String get(String prefixKey, String keyId) {
		String value = null;
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		String metricsName = "[Method-get] " + prefixKey;
		Transaction t = this.newTransaction(metricsName);
		try {
			jedis = pool.getResource();
			value = jedis.get(key);
			t.addData(key, value);
			if (value == null)
				Cat.logEvent(catMetricsType, metricsName + ":missed");// 未命中
			t.setStatus(Message.SUCCESS);
			logger.debug("获取缓存[JRedisProxy.get].正常：key=" + key + ",value=" + value);
		} catch (Exception e) {
			t.setStatus(e);
			logger.error("获取缓存[JRedisProxy.get].异常：key=" + key + ",value=" + value, e);
		} finally {
			if (jedis != null) {
				// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
				//pool.returnBrokenResource(jedis);
				jedis.close();
			}
			t.complete();
		}
		return value;
	}

	/**
	 * 批量获取缓存
	 * 
	 * @param keys 键
	 * @return
	 */
	public List<String> mget(String... keys) {
		return this.mget(null, keys);
	}

	/**
	 * 批量获取缓存
	 * 
	 * @param prefixKey 前缀键
	 * @param keyIds 键Id
	 * @return
	 */
	public List<String> mget(String prefixKey, String... keyIds) {
		List<String> value = null;
		Jedis jedis = null;
		String[] keys = this.getKey(prefixKey, keyIds);
		String metricsName = "[Method-mget] " + prefixKey;
		Transaction t = this.newTransaction(metricsName);
		try {
			jedis = pool.getResource();
			value = jedis.mget(keys);
			t.addData(keys + "", value);
			t.setStatus(Message.SUCCESS);
			logger.debug("批量获取缓存[JRedisProxy.mget].正常：key=" + keys + ",value=" + value);
		} catch (Exception e) {
			t.setStatus(e);
			logger.error("批量获取缓存[JRedisProxy.mget].异常：key=" + keys + ",value=" + value, e);
		} finally {
			if (jedis != null) {
				// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
				//pool.returnBrokenResource(jedis);
				jedis.close();
			}
			t.complete();
		}
		return value;
	}

	/**
	 * 删除缓存
	 * 
	 * @param key
	 */
	public void del(String key) {
		this.del(key, null);
	}

	/**
	 * 删除缓存
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 */
	public void del(String prefixKey, String keyId) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		String metricsName = "[Method-del] " + prefixKey;
		Transaction t = this.newTransaction(metricsName);
		try {
			jedis = pool.getResource();
			jedis.del(key);
			t.addData("DelKey", key);
			t.setStatus(Message.SUCCESS);
			logger.debug("删除缓存[JRedisProxy.del].正常：key=" + key);
		} catch (Exception e) {
			t.setStatus(e);
			logger.error("删除缓存[JRedisProxy.del].异常：key=" + key, e);
		} finally {
			if (jedis != null) {
				// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
				//pool.returnBrokenResource(jedis);
				jedis.close();
			}
			t.complete();
		}
	}
	

	/**
	 * 设置缓存过期时间
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param seconds 过期时间，单位秒
	 */
	public void expire(String prefixKey, String keyId, int seconds) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		String metricsName = "[Method-expire] " + prefixKey;
		Transaction t = this.newTransaction(metricsName);
		try {
			jedis = pool.getResource();
			jedis.expire(key, seconds);
			t.addData("expireKey", key);
			t.setStatus(Message.SUCCESS);
			logger.debug("设置缓存过期时间[JRedisProxy.expire].正常：key=" + key);
		} catch (Exception e) {
			t.setStatus(e);
			logger.error("设置缓存过期时间[JRedisProxy.expire].异常：key=" + key, e);
		} finally {
			if (jedis != null) {
				// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
				//pool.returnBrokenResource(jedis);
				jedis.close();
			}
			t.complete();
		}
	}

	/**
	 * 缓存是否存在
	 * 
	 * @param key
	 * @return
	 */
	public boolean exists(String key) {
		return this.exists(key, null);
	}

	/**
	 * 缓存是否存在
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @return
	 */
	public boolean exists(String prefixKey, String keyId) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		String metricsName = "[Method-exists] " + prefixKey;
		Transaction t = this.newTransaction(metricsName);
		try {
			jedis = pool.getResource();
			long ttl = jedis.ttl(key);
			t.addData("ExistsKey", key);
			t.setStatus(Message.SUCCESS);
			logger.debug("缓存是否存在[JRedisProxy.exists].正常：key=" + key);
			if (ttl > 0) {
				return true;
			} else {
				return false;
			}
		} catch (Exception e) {
			t.setStatus(e);
			logger.error("缓存是否存在[JRedisProxy.exists].异常：key=" + key, e);
		} finally {
			if (jedis != null) {
				// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
				//pool.returnBrokenResource(jedis);
				jedis.close();
			}
			t.complete();
		}
		return false;
	}
	
	/**
	 * 获取Key列表
	 * 
	 * @param pattern
	 * @return
	 */
	public Set<String> keys(String pattern) {
		Jedis jedis = null;
		Set<String> result = null;
		String metricsName = "[Method-keys] " + pattern;
		Transaction t = this.newTransaction(metricsName);
		try {
			jedis = pool.getResource();
			result = jedis.keys(pattern);
			t.addData("pattern", pattern);
			t.setStatus(Message.SUCCESS);
			logger.debug("获取Key列表[JRedisProxy.keys].正常：pattern=" + pattern);
		} catch (Exception e) {
			t.setStatus(e);
			logger.error("获取Key列表[JRedisProxy.keys].异常：pattern=" + pattern, e);
		} finally {
			if (jedis != null) {
				jedis.close();
			}
			t.complete();
		}
		return result;
	}
	

	/**
	 * zadd设置缓存
	 * @param prefixKey
	 * @param keyId
	 * @param score 排序分值
	 * @param member 元素
	 * @return
	 */
	public boolean zadd(String prefixKey, String keyId, long score, String member) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(member)) {
			String metricsName = "[Method-zadd] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, "score=" + score + ",member=" + member);
			try {
				jedis = pool.getResource();
				jedis.zadd(key, score, member);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.zadd].正常：key=" + key + ",score=" + score + ",member=" + member);
				return true;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.zadd].异常：key=" + key + ",score=" + score + ",member=" + member, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return false;
	}
	


	/**
	 * zrem删除元素
	 * @param prefixKey
	 * @param keyId
	 * @param members 元素集合
	 * @return
	 */
	public boolean zrem(String prefixKey, String keyId, String... members) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key) && members != null) {
			String metricsName = "[Method-zrem] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, "members=" + members);
			try {
				jedis = pool.getResource();
				jedis.zrem(key, members);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.zrem].正常：key=" + key + ",members=" + members);
				return true;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.zrem].异常：key=" + key + ",members=" + members, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return false;
	}
	

	/**
	 * zremrangeByScore删除元素集合
	 * 移除有续集中的member，移除member的score大于等于min小于等于max；返回移除元素个数
	 * @param prefixKey
	 * @param keyId
	 * @param members 元素集合
	 * @return
	 */
	public boolean zremrangeByScore(String prefixKey, String keyId, long min, long max) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key)) {
			String metricsName = "[Method-zremrangeByScore] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, "min=" + min + ",max=" + max);
			try {
				jedis = pool.getResource();
				jedis.zremrangeByScore(key, min, max);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.zremrangeByScore].正常：key=" + key  + ",min=" + min + ",max=" + max);
				return true;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.zremrangeByScore].异常：key=" + key  + ",min=" + min + ",max=" + max, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return false;
	}
	

	/**
	 * 
	 * zremrangebyrank删除元素集合
	 * 移除有续集中指定排名范围【start stop】的元素，返回移除元素个数
	 * @param prefixKey
	 * @param keyId
	 * @param start
	 * @param stop
	 * @return
	 */
	public boolean zremrangebyrank(String prefixKey, String keyId, long start, long stop) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key)) {
			String metricsName = "[Method-zremrangebyrank] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, "min=" + start + ",max=" + stop);
			try {
				jedis = pool.getResource();
				jedis.zremrangeByRank(key, start, stop);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.zremrangebyrank].正常：key=" + key  + ",min=" + start + ",max=" + stop);
				return true;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.zremrangebyrank].异常：key=" + key  + ",min=" + start + ",max=" + stop, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return false;
	}
	

	/**
	 * zcount获取缓存
	 * 返回有续集key中，score大于等于min并且小于等于max的member。返回结果按照score递增的次序排列。
	 * 可选withscores决定返回结果集中仅仅返回member还是返回member和score；可选参数limit 指定返回结果的数量区间。
	 * @param prefixKey
	 * @param keyId
	 * @param min
	 * @param max
	 * @param offset
	 * @param count
	 * @return
	 */
	public long zcount(String prefixKey, String keyId, double min, double max) {
		long result = 0;
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key)) {
			String metricsName = "[Method-zcount] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, "min="+min + ",max=" + max + ",min=" + min + ",max=" + max);
			try {
				jedis = pool.getResource();
				result = jedis.zcount(key, min, max);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.zcount].正常：key=" + key + ",min=" + min + ",max=" + max + ",min=" + min + ",max=" + max);
				return result;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.zcount].异常：key=" + key + ",min=" + min + ",max=" + max + ",min=" + min + ",max=" + max, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return result;
	}

	
	/**
	 * zrange获取缓存
	 * 返回有续集key中，score大于等于min并且小于等于max的member。返回结果按照score递增的次序排列。
	 * 可选withscores决定返回结果集中仅仅返回member还是返回member和score；可选参数limit 指定返回结果的数量区间。
	 * @param prefixKey
	 * @param keyId
	 * @param min
	 * @param max
	 * @param offset
	 * @param count
	 * @return
	 */
	public Set<String> zrange(String prefixKey, String keyId, long min, long max) {
		Set<String> result = new HashSet<String>();
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key)) {
			String metricsName = "[Method-zrange] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, "min="+min + ",max=" + max + ",min=" + min + ",max=" + max);
			try {
				jedis = pool.getResource();
				result = jedis.zrange(key, min, max);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.zrange].正常：key=" + key + ",min=" + min + ",max=" + max + ",min=" + min + ",max=" + max);
				return result;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.zrange].异常：key=" + key + ",min=" + min + ",max=" + max + ",min=" + min + ",max=" + max, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return result;
	}

	/**
	 * zrangeByScore获取缓存
	 * 返回有续集key中，score大于等于min并且小于等于max的member。返回结果按照score递增的次序排列。
	 * 可选withscores决定返回结果集中仅仅返回member还是返回member和score；可选参数limit 指定返回结果的数量区间。
	 * @param prefixKey
	 * @param keyId
	 * @param min
	 * @param max
	 * @param offset
	 * @param count
	 * @return
	 */
	public Set<String> zrangeByScore(String prefixKey, String keyId, double min, double max, int offset, int count) {
		Set<String> result = new HashSet<String>();
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key)) {
			String metricsName = "[Method-zrangeByScore] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, "min="+min + ",max=" + max + ",offset=" + offset + ",count=" + count);
			try {
				jedis = pool.getResource();
				result = jedis.zrangeByScore(key, min, max, offset, count);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.zrangeByScore].正常：key=" + key + ",min=" + min + ",max=" + max + ",offset=" + offset + ",count=" + count);
				return result;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.zrangeByScore].异常：key=" + key + ",min=" + min + ",max=" + max + ",offset=" + offset + ",count=" + count, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return result;
	}
	
	/**
	 * 在过期时间内对Key进行累加
	 * 
	 * @param key 键
	 * @param expireSecond 过期时间
	 * @return
	 */
	public int incrAndRtnKeyTimes(String key, int expireSecond) {
		return this.incrAndRtnKeyTimes(key, null, expireSecond);
	}

	/**
	 * 在过期时间内对Key进行累加
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param expireSecond 过期时间
	 * @return
	 */
	public int incrAndRtnKeyTimes(String prefixKey, String keyId, int expireSecond) {
		int value = 0;
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key)) {
			String metricsName = "[Method-incrAndRtnKeyTimes] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, "expireSecond=" + expireSecond);
			try {
				jedis = pool.getResource();
				jedis.incr(key);
				jedis.expire(key, expireSecond);
				value = Integer.valueOf(jedis.get(key));
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.incrAndRtnKeyTimes].正常：key=" + key + ",expireSecond=" + expireSecond + ",value=" + value);
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.incrAndRtnKeyTimes].正常：key=" + key + ",expireSecond=" + expireSecond + ",value=" + value, e);
			} finally {
				if (jedis != null) {
					jedis.close();
				}
				t.complete();
			}
		}
		return value;
	}
	

	/**
	 * 获取hash缓存
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param field hash key
	 * @return
	 */
	public String hget(String prefixKey, String keyId, String field) {
		String value = null;
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		String metricsName = "[Method-hget] " + prefixKey;
		Transaction t = this.newTransaction(metricsName);
		try {
			jedis = pool.getResource();
			value = jedis.hget(key, field);
			t.addData(key + "_" + field, value);
			if (value == null)
				Cat.logEvent(catMetricsType, metricsName + ":missed");// 未命中
			t.setStatus(Message.SUCCESS);
			logger.debug("获取缓存[JRedisProxy.hget].正常：key=" + key + ",field=" + field + ",value=" + value);
		} catch (Exception e) {
			t.setStatus(e);
			logger.error("获取缓存[JRedisProxy.hget].异常：key=" + key + ",field=" + field + ",value=" + value, e);
		} finally {
			if (jedis != null) {
				// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
				//pool.returnBrokenResource(jedis);
				jedis.close();
			}
			t.complete();
		}
		return value;
	}

	/**
	 * 获取hash缓存
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param field hash key
	 * @return
	 */
	public Map<String,String> hgetAll(String prefixKey, String keyId) {
		Map<String,String> value = null;
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		String metricsName = "[Method-hgetAll] " + prefixKey;
		Transaction t = this.newTransaction(metricsName);
		try {
			jedis = pool.getResource();
			value = jedis.hgetAll(key);
			t.addData(key, value);
			if (value == null)
				Cat.logEvent(catMetricsType, metricsName + ":missed");// 未命中
			t.setStatus(Message.SUCCESS);
			logger.debug("获取缓存[JRedisProxy.hgetAll].正常：key=" + key + ",value=" + value);
		} catch (Exception e) {
			t.setStatus(e);
			logger.error("获取缓存[JRedisProxy.hgetAll].异常：key=" + key + ",value=" + value, e);
		} finally {
			if (jedis != null) {
				// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
				//pool.returnBrokenResource(jedis);
				jedis.close();
			}
			t.complete();
		}
		return value;
	}

	/**
	 * 设置hash缓存
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param field hash key
	 * @param value 值
	 * @param seconds 过期时间
	 * @return
	 */
	public boolean hset(String prefixKey, String keyId, String field, String value) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(field) && StringUtils.isNotBlank(value)) {
			String metricsName = "[Method-hset] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key + "_" + field, value);
			try {
				jedis = pool.getResource();
				jedis.hset(key, field, value);
				t.setStatus(Message.SUCCESS);
				logger.debug("设置缓存[JRedisProxy.hset].正常：key=" + key + ",field=" + field + ",value=" + value);
				return true;
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.hset].异常：key=" + key + ",field=" + field + ",value=" + value, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return false;
	}
	
	/**
	 * 删除hash缓存
	 * 
	 * @param prefixKey 前缀键
	 * @param keyId 键Id
	 * @param field hash key
	 * @param value 值
	 * @param seconds 过期时间
	 * @return
	 */
	public boolean hdel(String prefixKey, String keyId, String field) {
		Jedis jedis = null;
		String key = this.getKey(prefixKey, keyId);
		if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(field)) {
			String metricsName = "[Method-hdel] " + prefixKey;
			Transaction t = this.newTransaction(metricsName);
			t.addData(key, field);
			try {
				jedis = pool.getResource();
				Long result = jedis.hdel(key, field);
				if(result == 1){
					t.setStatus(Message.SUCCESS);
					logger.debug("设置缓存[JRedisProxy.hdel].正常：key=" + key + ",field=" + field);
					return true;
				}else{
					t.setStatus("failed");
					logger.debug("设置缓存[JRedisProxy.hdel].失败：key=" + key + ",field=" + field + ",result=" + result);
					return false;
				}
			} catch (Exception e) {
				t.setStatus(e);
				logger.error("设置缓存[JRedisProxy.hdel].异常：key=" + key + ",field=" + field, e);
			} finally {
				if (jedis != null) {
					// jedis版本升级 2.1.0 =》 2.9.0 方法变更 pool.returnBrokenResource => jedis.close();
					//pool.returnBrokenResource(jedis);
					jedis.close();
				}
				t.complete();
			}
		}
		return false;
	}
}
