/**
 * 
 */
package com.xunlei.netty;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.concurrent.ExecutorService;

import javax.annotation.Resource;

import org.jboss.netty.bootstrap.ServerBootstrap;
import org.jboss.netty.channel.ChannelPipelineFactory;
import org.jboss.netty.channel.socket.nio.NioServerSocketChannelFactory;
import org.jboss.netty.logging.InternalLoggerFactory;
import org.jboss.netty.logging.Slf4JLoggerFactory;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import com.xunlei.netty.consul.ConsulFactory;
import com.xunlei.netty.grpcserver.GRPCServerFactory;
import com.xunlei.netty.httpserver.HttpServerPipelineFactory;
import com.xunlei.netty.httpserver.component.AbstractPageDispatcher;
import com.xunlei.netty.soaserver.SOAServerPipelineFactory;
import com.xunlei.netty.soaserver.component.BaseSOAPageDispatcher;
import com.xunlei.netty.util.NettyServerConfig;
import com.xunlei.netty.util.IPAuthenticator;
import com.xunlei.netty.util.Log;
import com.xunlei.netty.util.NetUtil;
import com.xunlei.netty.util.NioSocketBootstrapFactory;
import com.xunlei.netty.util.spring.BeanUtil;
import com.xunlei.netty.util.spring.SpringBootstrap;

/**
 * 引导程序
 * 
 * @author wangcanyi
 *
 */
@Component
public class BootstrapFactory {
	private static final Logger log = Log.getLogger();
	public static ApplicationContext CONTEXT;
	/**
	 * 配置信息
	 */
	@Autowired
	private NettyServerConfig config;
	/**
	 * 初始化处理接口
	 */
	private Runnable initialRunnable = null;
	/**
	 * 关闭处理接口
	 */
	private Runnable shutdownRunanble = null;
	/**
	 * HTTP 管道处理
	 */
	@Autowired
	private AbstractPageDispatcher pageDispatcher;
	/**
	 * SOA管道处理
	 */
	@Autowired
	private BaseSOAPageDispatcher soaBasePageDispatcher;

	/**
	 * 服务管道工厂
	 */
	private NioServerSocketChannelFactory serverSocketChannelFactory;
	/**
	 * HTTP管道
	 */
	@Resource
	private HttpServerPipelineFactory httpServerPipelineFactory;
	/**
	 * SOA管道
	 */
	@Resource
	private SOAServerPipelineFactory soaServerPipelineFactory;
	/**
	 * GRPC服务工厂
	 */
	@Resource
	private GRPCServerFactory grpcServerFactory;

	/**
	 * 引导程序
	 * 
	 * @param args 程序参数
	 * @param initialRunnable 初始化处理接口
	 * @param shutdownRunnalbe 关闭处理接口
	 * @param springConfigLocations spring配置文件
	 * @return
	 * @throws IOException
	 */
	public static ApplicationContext main(String[] args, Runnable initialRunnable, Runnable shutdownRunnalbe, String... springConfigLocations) throws Exception {
		log.debug("Netty服务.引导程序.开始");
		try {
			long before = System.currentTimeMillis();
			CONTEXT = SpringBootstrap.load(springConfigLocations);
			System.err.println("----->loadContext      [" + (System.currentTimeMillis() - before) + "MS]");
		} catch (Throwable t) {
			t.printStackTrace();
			System.exit(1); // 遇到任何异常直接退出
		}
		BootstrapFactory bootstrapFactory = BeanUtil.getTypedBean("bootstrapFactory");
		bootstrapFactory.initialRunnable = initialRunnable;
		bootstrapFactory.shutdownRunanble = shutdownRunnalbe;
		if (isArgWhat(args, "stop", "shutdown")) {
			// 等待实现
			System.exit(0);
		} else {
			log.debug("Netty服务.服务启动.开始");
			bootstrapFactory.start();
			log.debug("Netty服务.服务启动.结束");
		}
		log.debug("Netty服务.引导程序.结束");
		return CONTEXT;
	}

	/**
	 * 引导程序
	 * 
	 * @param args 程序参数
	 * @param initialRunnable 初始化处理接口
	 * @param springConfigLocations spring配置文件
	 * @return
	 * @throws IOException
	 */
	public static ApplicationContext main(String[] args, Runnable initialRunnable, String... springConfigLocations) throws Exception {
		return main(args, initialRunnable, null, springConfigLocations);
	}

	/**
	 * 引导程序
	 * 
	 * @param args 程序参数
	 * @param springConfigLocations spring配置文件
	 * @return
	 * @throws IOException
	 */
	public static ApplicationContext main(String[] args, String... springConfigLocations) throws Exception {
		return main(args, null, null, springConfigLocations);
	}

	/**
	 * 开始启动
	 * 
	 * @throws Exception
	 */
	private void start() throws Exception {
		try {
			initEnv();// 初始化环境信息
			initOutter();// 初始化外部业务逻辑
			initHttpServer();// 初始化HTTP服务
			initSOAServer();// 初始化SOA服务
			initGRPCServer();//初始化GRPC服务
		} catch (Throwable t) {
			t.printStackTrace();
			System.exit(-1);
		}
	}

	/**
	 * 初始化环境信息
	 * @throws Exception 
	 */
	private void initEnv() throws Exception {
		InternalLoggerFactory.setDefaultFactory(new Slf4JLoggerFactory());
		IPAuthenticator.setBusinessWhiteIp(config.getIpWhiteList());// 设置Ip白名单
		IPAuthenticator.reload();// 加载配置
		if(config.isConsulIsEnable()){//是否启用Consul
			String serviceName = config.getConsulServiceName();
			int httpPort = config.getListen_port();
			int grpcPort = config.getListen_grpc_port();
			ConsulFactory.getInstance().registerService(serviceName, httpPort, grpcPort);
		}
		// 初始化管道工厂
		serverSocketChannelFactory = new NioServerSocketChannelFactory(NettyServerConfig.bossExecutor, NettyServerConfig.workerExecutor,
				config.getRealWorkerCount());
	}

	/**
	 * 初始化外部逻辑
	 */
	private void initOutter() {
		if (this.initialRunnable != null) {
			long before = System.currentTimeMillis();
			initialRunnable.run();
			System.err.println("----->initialRunnable  [" + (System.currentTimeMillis() - before) + "MS]");
		}
	}

	/**
	 * 初始化Http服务
	 */
	private void initHttpServer() throws Exception {
		int port = config.getListen_port();// HTTP监听端口
		try {
			if (port > 0) {
				long beginBind = System.currentTimeMillis();
				pageDispatcher.init();// 初始化业务接口
				ServerBootstrap serverBootstrap = new ServerBootstrap(serverSocketChannelFactory);
				serverBootstrap.setPipelineFactory(httpServerPipelineFactory.rebuildPipeline());// Set up the event pipeline factory.
				NetUtil.checkSocketPortBind(port);// 校验端口是否被占用
				bindAndBootstrapOptions(port, serverBootstrap);// 绑定端口
				System.err.println("----->bindHttpPort   [" + port + "]  [" + (System.currentTimeMillis() - beginBind) + "MS]");
				System.err.println("HTTP服务器启动完毕.(port[" + port + "])");
			}
		} catch (Exception e) {
			System.err.println("HTTP服务器启动失败.(port[" + port + "]) ERROR:" + e.getMessage());
			throw e;
		}
	}

	/**
	 * 初始化SOA服务
	 * 
	 * @throws Exception
	 */
	private void initSOAServer() throws Exception {
		int soaPort = config.getListen_soa_port();
		try {
			if (soaPort > 0) {
				long beginBind = System.currentTimeMillis();
				soaBasePageDispatcher.init();// 初始化SOA业务接口
				ServerBootstrap serverBootstrap = new ServerBootstrap(serverSocketChannelFactory);
				serverBootstrap.setPipelineFactory(soaServerPipelineFactory.rebuildPipeline());// Set up the event pipeline factory.
				NetUtil.checkSocketPortBind(soaPort);// 校验端口是否被占用
				bindAndBootstrapOptions(soaPort, serverBootstrap);// 绑定端口
				System.err.println("----->bindSOAPort   [" + soaPort + "]  [" + (System.currentTimeMillis() - beginBind) + "MS]");
				System.err.println("SOA服务器启动完毕.(port[" + soaPort + "])");
			}
		} catch (Exception e) {
			System.err.println("SOA服务器启动失败.(port[" + soaPort + "]) ERROR:" + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * 初始化GRPC服务
	 * @throws Exception
	 */
	private void initGRPCServer() throws Exception {
		int grpcPort = config.getListen_grpc_port();
		try {
			if (grpcPort > 0) {
				long beginBind = System.currentTimeMillis();
				grpcServerFactory.start();
				System.err.println("----->bindGRPCPort   [" + grpcPort + "]  [" + (System.currentTimeMillis() - beginBind) + "MS]");
				System.err.println("GRPC服务器启动完毕.(port[" + grpcPort + "])");
			}
		} catch (Exception e) {
			System.err.println("GRPC服务器启动失败.(port[" + grpcPort + "]) ERROR:" + e.getMessage());
			throw e;
		}
	}

	/**
	 * 绑定端口
	 * 
	 * @param port
	 * @param serverBootstrap
	 */
	private void bindAndBootstrapOptions(int port, ServerBootstrap serverBootstrap) {
		if (port > 0) {
			NioSocketBootstrapFactory.setBootstrapOptions(serverBootstrap, config.getConnectTimeoutMillis(), config.getReceiveBufferSize(),
					config.getSendBufferSize());// 设置参数
			serverBootstrap.bind(new InetSocketAddress(port));// Bind and start to accept incoming connections.
		}
	}

	/**
	 * args里是否含有参数
	 * 
	 * @param args
	 * @param what
	 * @return
	 */
	private static boolean isArgWhat(String[] args, String... what) {
		if (args == null || args.length == 0) {
			return false;
		}
		String arg = args[0].toLowerCase();
		for (String w : what) {
			if (arg.indexOf(w) >= 0) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 是否正在关闭，当前未实现
	 * 
	 * @return
	 */
	public boolean isStop() {
		return false;
	}
}
