package com.xunlei.netty.util;

import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;

import com.xunlei.netty.cache.JRedisProxy;
import com.xunlei.util.Log;

/**
 * 基于redis的setnx、get、getset的分布式锁
 * <p>
 * 注意：该实现方式适用于版本低于2.6.12的redis。如果redis版本大于等于2.6.12，则最好使用
 * <code>com.xunlei.niux.center.util.RedisLock2Util</code>这个类
 * 
 * @author HuangGuangHao
 *
 */
public class RedisLockUtil {
	
	private static Logger logger = Log.getLogger(RedisLockUtil.class.getName());
	
	// 单个分布式锁的超时时间(单位：毫秒)
    private static final long DEFAULT_SINGLE_EXPIRE_TIME = 3000L;
    // 重试等待时长(单位：毫秒)
    private static final int DEFAULT_RETRY_WATING_TIME = 300;
    // redis服务器的失败重连次数
    private static final int RETRY_COUNT = 5;
    
    /**
     * 获取锁
     * 
     * @param key 键
     * @return 成功时返回true；否则返回false。
     */
    public static boolean tryLock(String key) {
    	return tryLock(key, -1L);
    }
    
    /**
     * 获取锁
     * 
     * @param key 键
     * @param connTimeout 连接超时时间(单位：毫秒)。如果该值小于等于0，则表示不重新尝试获取锁。
     * @return 成功时返回true；否则返回false。
     */
    public static boolean tryLock(String key, long connTimeout) {
    	return tryLock(key, connTimeout, TimeUnit.MILLISECONDS);
    }
	
    /**
     * 获取锁
     * 
     * @param key 键
     * @param connTimeout 连接超时时间
     * @param connTimeoutUnit 连接超时时间的单位。如果该值小于等于0，则表示不重新尝试获取锁。
     * @return 成功时返回true；否则返回false。
     */
	public static boolean tryLock(String key, long connTimeout, TimeUnit connTimeoutUnit) {
		return tryLock(key, connTimeout, connTimeoutUnit, DEFAULT_SINGLE_EXPIRE_TIME, TimeUnit.MILLISECONDS);
    }
	
	/**
	 * 获取锁
	 * 
	 * @param key 键
	 * @param connTimeout 连接超时时间。如果该值小于等于0，则表示不重新尝试获取锁。
	 * @param connTimeoutUnit 连接超时时间的单位
	 * @param keyTimeout 键超时时间
	 * @param keyTimeoutUnit 键超时时间的单位
	 * @return 成功时返回true；否则返回false。
	 */
	public static boolean tryLock(String key, long connTimeout, TimeUnit connTimeoutUnit, long keyTimeout, TimeUnit keyTimeoutUnit) {
		if(key == null) {
			return false;
		}
		
		boolean result = false;
		long originTime = System.currentTimeMillis();
		connTimeout = connTimeout > 0 ? connTimeoutUnit.toMillis(connTimeout) : connTimeout;
		keyTimeout = keyTimeoutUnit.toMillis(keyTimeout);
		
		try {
			do {
				result = getLock(key, keyTimeout);
				if (result || connTimeout <= 0) {
					break;
				}
				Thread.sleep(DEFAULT_RETRY_WATING_TIME);
			} while ((System.currentTimeMillis() - originTime) < connTimeout);
			
		} catch (InterruptedException e) {
			logger.error("[RedisLockUtil.tryLock] key={},timeout={},unit={} Exception: ",
					new Object[] { key, connTimeout, connTimeoutUnit, e });
		}
		
		return result;
	}

	private static boolean getLock(String key, long keyTimeout) {
		Boolean result = null;
		JRedisProxy proxy = JRedisProxy.getInstance();
		for(int i = 0; i < RETRY_COUNT; i++) {
			result = proxy.setnx(key, String.valueOf(System.currentTimeMillis() + keyTimeout));
			if(result != null) {
				break;
			}
		}
		
		if(result == null) {
			// 连接异常
			return false;
		}
		
		if (result) {
			// 成功获取到分布式锁
			return true;
		}
		
		// 分布式锁已经被其它线程所持有
		
		// 判断分布式锁是否过期
		String oldExpiredTimeStr = proxy.get(key);
		if(oldExpiredTimeStr != null) {
			long oldExpiredTime = Long.parseLong(oldExpiredTimeStr);
			long now = System.currentTimeMillis();
			if(now >= oldExpiredTime) {
				// 分布式锁已过期
				long currentExpiredTime = Long.parseLong(proxy.getSet(key, String.valueOf(now + keyTimeout)));
				if(currentExpiredTime == oldExpiredTime) {
					// 成功获取到分布式锁
					return true;
				}
			}
		}
		
		return false;
	}

	/**
	 * 释放锁
	 * 
	 * @param key 键
	 */
    public static void unLock(String key) {
    	if(key == null) {
    		return;
    	}
    	JRedisProxy.getInstance().del(key);
    }
    
}
