package com.xunlei.netty.util;

import java.util.UUID;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;

import com.xunlei.netty.cache.JRedisProxy;
import com.xunlei.util.Log;

/**
 * 基于redis的{@literal SET <key> <value> PX <milliseconds> NX}的分布式锁
 * <p>
 * 注意：如果redis版本小于2.6.12，则不要使用该类，而是要使用
 * <code>com.xunlei.niux.center.util.RedisLockUtil</code>这个类
 * 
 * @see <a href="http://redis.io/commands/set">redis set命令</a>
 * @see <a href="http://redis.io/topics/distlock#correct-implementation-with-a-single-instance">使用redis实现分布式锁的官方文档</a>
 * 
 * @author HuangGuangHao
 *
 */
public class RedisLock2Util {
	
	private static Logger logger = Log.getLogger(RedisLock2Util.class.getName());
	
	// 单个分布式锁的超时时间(单位：秒)
    private static final int DEFAULT_SINGLE_EXPIRE_TIME = 3;
    // 重试等待时长(单位：毫秒)
    private static final int DEFAULT_RETRY_WATING_TIME = 300;
    
    private static final String STATUS_OK = "OK";
    
    /**
     * 获取锁
     * 
     * @param key 键
     * @return 成功时返回锁ID；否则返回null。
     */
    public static String tryLock(String key) {
    	return tryLock(key, -1L);
    }
    
    /**
     * 获取锁
     * 
     * @param key 键
     * @param connTimeout 连接超时时间(单位：毫秒)。如果该值小于等于0，则表示不重新尝试获取锁。
     * @return 成功时返回锁ID；否则返回null。
     */
    public static String tryLock(String key, long connTimeout) {
    	return tryLock(key, connTimeout, TimeUnit.MILLISECONDS);
    }
	
    /**
     * 获取锁
     * 
     * @param key 键
     * @param connTimeout 连接超时时间
     * @param connTimeoutUnit 连接超时时间的单位。如果该值小于等于0，则表示不重新尝试获取锁。
     * @return 成功时返回锁ID；否则返回null。
     */
	public static String tryLock(String key, long connTimeout, TimeUnit connTimeoutUnit) {
		return tryLock(key, connTimeout, connTimeoutUnit, DEFAULT_SINGLE_EXPIRE_TIME);
    }
	
	/**
	 * 获取锁
	 * 
	 * @param key 键
	 * @param connTimeout 连接超时时间。如果该值小于等于0，则表示不重新尝试获取锁。
	 * @param connTimeoutUnit 连接超时时间的单位
	 * @param keyTimeoutInSeconds 键超时时间(单位：秒)
	 * @return 成功时返回锁ID；否则返回null。
	 */
	public static String tryLock(String key, long connTimeout, TimeUnit connTimeoutUnit, int keyTimeoutInSeconds) {
		if(key == null) {
			return null;
		}
		
		long originTime = System.currentTimeMillis();
		connTimeout = connTimeout > 0 ? connTimeoutUnit.toMillis(connTimeout) : connTimeout;
		
		try {
			do {
				String lockId = getNextLockId();
				String code = set(key, lockId, keyTimeoutInSeconds);

				if(isSucc(code)) {
					return lockId;
				}
				
				if(connTimeout <= 0) {
					break;
				}
				
				Thread.sleep(DEFAULT_RETRY_WATING_TIME);
			} while ((System.currentTimeMillis() - originTime) < connTimeout);
			
		} catch (InterruptedException e) {
			logger.error("[RedisLockUtil.tryLock] key={},timeout={},unit={} Exception: ",
					new Object[] { key, connTimeout, connTimeoutUnit, e });
		}
		
		return null;
	}

	/**
     * 释放锁
     * 
     * @param key 键
     * @param lockId 锁ID
     */
    public static void unLock(String key, String lockId) {
    	if(key == null || lockId == null) {
    		return;
    	}
        
    	String currentLockId = JRedisProxy.getInstance().get(key);
    	if(lockId.equals(currentLockId)) {
    		JRedisProxy.getInstance().del(key);
    	}
    }
    
    private static String getNextLockId() {
    	return UUID.randomUUID().toString();
    }
    
    private static String set(String key, String lockId, int keyTimeoutInSeconds) {
    	return JRedisProxy.getInstance().set(key, lockId, "NX", "EX", keyTimeoutInSeconds);
    }
    
    private static boolean isSucc(String code) {
    	return STATUS_OK.equalsIgnoreCase(code);
    }
    
}
