/**
 * 
 */
package com.xunlei.netty.soaserver.component;

import java.io.IOException;
import java.util.Properties;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;

import com.xunlei.netty.util.Log;

/**
 * SOA服务基类
 * 
 * @author wangcanyi
 *
 */
public class SOAServiceBase {
	private static final Logger log = Log.getLogger();
	private String appName;// 应用领域名/项目名
	private String serviceHost;// 服务地址Host
	private int servicePort;// 服务地址端口
	private int connectionTimeout = 2000;// 超时时间，毫秒数
	private String signKey;// 签名Key
	private boolean consulIsEnable = false;// Consul是否启用
	private String grpcServiceHost;//GRPC服务地址
	private int grpcServicePort;//GRPC服务端口

	/**
	 * 构造器
	 * 
	 * @param appName 项目名
	 * @param configUrl 配置文件地址（ServiceHost，ServicePort，ConnectionTimeout）
	 */
	public SOAServiceBase(String appName, String configUrl) {
		this.appName = appName;
		Properties config = this.getServiceClientProperties(configUrl);
		this.serviceHost = config.getProperty("ServiceHost", "");
		this.servicePort = Integer.parseInt(config.getProperty("ServicePort", "0"));
		this.connectionTimeout = Integer.parseInt(config.getProperty("ConnectionTimeout", "2000"));
		this.signKey = config.getProperty("SignKey", "");
		this.consulIsEnable = Boolean.parseBoolean(config.getProperty("ConsulIsEnable", "false"));
		this.grpcServiceHost = config.getProperty("GRPCServiceHost", "");
		this.grpcServicePort = Integer.parseInt(config.getProperty("GRPCServicePort", "0"));
	}

	/**
	 * 构造器
	 * 
	 * @param serviceBase 服务基类
	 */
	public SOAServiceBase(SOAServiceBase serviceBase) {
		this.appName = serviceBase.getAppName();
		this.serviceHost = serviceBase.getServiceHost();
		this.servicePort = serviceBase.getServicePort();
		this.connectionTimeout = serviceBase.getConnectionTimeout();
		this.signKey = serviceBase.getSignKey();
		this.consulIsEnable = serviceBase.isConsulIsEnable();
		this.grpcServiceHost = serviceBase.getGrpcServiceHost();
		this.grpcServicePort = serviceBase.getGrpcServicePort();
	}

	/**
	 * 获取服务客户端配置
	 * 
	 * @return
	 */
	private Properties getServiceClientProperties(String configUrl) {
		Properties config = new Properties();
		if (StringUtils.isNotBlank(configUrl)) {
			try {
				config.load(SOAServiceBase.class.getClassLoader().getResourceAsStream(configUrl));
				log.debug("获取服务客户端配置[SOAServiceBase.getServiceClientProperties].正常：configUrl" + configUrl);
			} catch (IOException e) {
				log.error("获取服务客户端配置[SOAServiceBase.getServiceClientProperties].异常：configUrl" + configUrl, e);
			}
		}
		return config;
	}

	/**
	 * @return 应用领域名/项目名
	 */
	public String getAppName() {
		return appName;
	}

	/**
	 * @param appName 应用领域名/项目名
	 */
	public void setAppName(String appName) {
		this.appName = appName;
	}

	/**
	 * @return 服务地址Host
	 */
	public String getServiceHost() {
		return serviceHost;
	}

	/**
	 * @param serviceHost 服务地址Host
	 */
	public void setServiceHost(String serviceHost) {
		this.serviceHost = serviceHost;
	}

	/**
	 * @return 服务地址端口
	 */
	public int getServicePort() {
		return servicePort;
	}

	/**
	 * @param servicePort 服务地址端口
	 */
	public void setServicePort(int servicePort) {
		this.servicePort = servicePort;
	}

	/**
	 * @return 超时时间，毫秒数
	 */
	public int getConnectionTimeout() {
		return connectionTimeout;
	}

	/**
	 * @param connectionTimeout 超时时间，毫秒数
	 */
	public void setConnectionTimeout(int connectionTimeout) {
		this.connectionTimeout = connectionTimeout;
	}

	/**
	 * @return 签名Key
	 */
	public String getSignKey() {
		return signKey;
	}

	/**
	 * @param signKey 签名Key
	 */
	public void setSignKey(String signKey) {
		this.signKey = signKey;
	}

	/**
	 * @return Consul是否启用
	 */
	public boolean isConsulIsEnable() {
		return consulIsEnable;
	}

	/**
	 * @param consulIsEnable Consul是否启用
	 */
	public void setConsulIsEnable(boolean consulIsEnable) {
		this.consulIsEnable = consulIsEnable;
	}

	/**
	 * @return GRPC服务地址
	 */
	public String getGrpcServiceHost() {
		return grpcServiceHost;
	}

	/**
	 * @param grpcServiceHost GRPC服务地址
	 */
	public void setGrpcServiceHost(String grpcServiceHost) {
		this.grpcServiceHost = grpcServiceHost;
	}

	/**
	 * @return GRPC服务端口
	 */
	public int getGrpcServicePort() {
		return grpcServicePort;
	}

	/**
	 * @param grpcServicePort GRPC服务端口
	 */
	public void setGrpcServicePort(int grpcServicePort) {
		this.grpcServicePort = grpcServicePort;
	}
}
