package com.xunlei.netty.httpserver.cmd;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.ResourceBundle;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xunlei.netty.httpserver.cmd.annotation.CmdAdmin;
import com.xunlei.netty.httpserver.component.TimeoutInterrupter;
import com.xunlei.netty.httpserver.component.XLContextAttachment;
import com.xunlei.netty.httpserver.component.XLHttpRequest;
import com.xunlei.netty.httpserver.component.XLHttpResponse;
import com.xunlei.netty.util.EmptyChecker;
import com.xunlei.netty.util.NettyServerConfig;
import com.xunlei.netty.util.Log;
import com.xunlei.netty.util.ResourceBundleUtil;
import com.xunlei.netty.util.ValueUtil;
import com.xunlei.netty.util.spring.BeanUtil;
import com.xunlei.netty.util.spring.Config;
import com.xunlei.netty.util.spring.SpringBootstrap;

/**
 * CmdMappers类
 * 
 * @author ZengDong
 * @since 2010-6-7 下午11:14:59
 */
@Component
public class CmdMappers {

	@Autowired
	private TimeoutInterrupter timeoutInterrupter;

	public static class CmdMeta {

		private BaseCmd cmd;
		private Method method;
		/** cmd所在的类名简称，专用于通配符配置超时，配置整个类下面的所有cmd的配置 */
		private String baseName;
		private String name;
		/**
		 * 是否管理员接口
		 */
		private boolean isAdmin = false;

		/**
		 * 每次业务操作时间,单位秒, <0 表示此命令直接Disable,0表示不超时,>0 指具体超时秒数
		 */
		private int timeout;

		public CmdMeta(BaseCmd cmd, Method method) {
			this.cmd = cmd;
			this.method = method;
			this.baseName = cmd.getClass().getSimpleName() + ".*";
			this.name = cmd.getClass().getSimpleName() + "." + method.getName();
			if (method.getAnnotation(CmdAdmin.class) != null)
				isAdmin = true;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}
			CmdMeta other = (CmdMeta) obj;
			if (cmd == null) {
				if (other.cmd != null) {
					return false;
				}
			} else if (!cmd.equals(other.cmd)) {
				return false;
			}
			if (method == null) {
				if (other.method != null) {
					return false;
				}
			} else if (!method.equals(other.method)) {
				return false;
			}
			return true;
		}

		public BaseCmd getCmd() {
			return cmd;
		}

		public Method getMethod() {
			return method;
		}

		/**
		 * 是否管理员接口
		 * 
		 * @return
		 */
		public boolean isAdmin() {
			return isAdmin;
		}

		public int getTimeout() {
			return timeout;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((cmd == null) ? 0 : cmd.hashCode());
			result = prime * result + ((method == null) ? 0 : method.hashCode());
			return result;
		}

		public boolean isDisable() {
			return timeout < 0;
		}

		@Override
		public String toString() {
			return name;
		}

		/**
		 * 获取CMD相关的描述信息，用于生成在线DOC
		 */
		public String getCmdDescription() {
			return method.toGenericString();
		}
	}

	private static final Logger log = Log.getLogger();

	/**
	 * 去除请求url的路径中最后一个"/"号
	 */
	private static String sanitizePath(String path) {
		if (StringUtils.isNotBlank(path)) {
			path = path.trim();
			int len = path.length();
			if (len > 1 && path.lastIndexOf('/') == len - 1) {
				path = path.substring(0, len - 1);
			}
		}
		return path;
	}

	/**
	 * 所有Urk映射CmdMeta
	 */
	private Map<String, CmdMeta> cmdAllMap = new LinkedHashMap<String, CmdMeta>();
	private final Map<CmdMeta, CmdMeta> cmdMetaUnite = new HashMap<CmdMeta, CmdMeta>();
	/**
	 * 所有spring中配置好且符合cmd类规范的对象
	 */
	private List<BaseCmd> cmds;// 所有spring中配置好且符合cmd类规范的对象
	@Autowired
	private NettyServerConfig config;
	/**
	 * 全局超时时间
	 */
	private int globalTimeout = 0;

	/**
	 * @return 所有Urk映射CmdMeta
	 */
	public Map<String, CmdMeta> getCmdAllMap() {
		return cmdAllMap;
	}

	/**
	 * 小写转换
	 * 
	 * @param name
	 * @return
	 */
	private String _lower(String name) {
		return name.substring(0, 1).toLowerCase() + name.substring(1);
	}

	/**
	 * 获取CMD类
	 * 
	 * @param name
	 * @return
	 */
	private BaseCmd getCmd(String name) {
		Object obj = BeanUtil.getTypedBean(name);
		if (obj instanceof BaseCmd) {
			Class<?> clazz = obj.getClass();
			if (clazz.getAnnotation(Deprecated.class) != null) {
				return null;
			}
			return (BaseCmd) obj;
		}
		return null;
	}

	/**
	 * 根据url地址，获取CmdMeta
	 * 
	 * @param path
	 * @return
	 * @throws Exception
	 */
	public CmdMeta getCmdMeta(String path) throws Exception {
		path = sanitizePath(path);
		CmdMeta meta = cmdAllMap.get(path);
		return meta;
	}

	/**
	 * 获取cmd显示名
	 * 
	 * @param cmd
	 * @return
	 */
	private String getCmdName(BaseCmd cmd) {
		String cmdSuffix = config.getCmdSuffix();
		String cmdName = _lower(cmd.getClass().getSimpleName());

		// 这里解决 因为cglib造成其类名被替换的情况
		// 2011-03-18 原来没有此bug的原因是因为原来是不先通过getCmds() 获得其BaseCmd再继续获得其className,而是直接通过 Bootstrap.CONTEXT.getBeanDefinitionNames()来找到的
		int idx = cmdName.indexOf('$');
		if (idx > 0) {
			cmdName = cmdName.substring(0, idx);
		}

		if (cmdName.endsWith(cmdSuffix)) {
			cmdName = cmdName.substring(0, cmdName.length() - cmdSuffix.length());
		}
		return cmdName;
	}

	/**
	 * 获取所有的CMD
	 * 
	 * @return
	 */
	private List<BaseCmd> getCmds() {
		if (cmds == null) {
			cmds = new ArrayList<BaseCmd>();
			for (String name : SpringBootstrap.getContext().getBeanDefinitionNames()) {
				BaseCmd cmd = getCmd(name);
				if (cmd != null) {
					cmds.add(cmd);
				}
			}
		}
		return cmds;
	}

	/**
	 * 获得一个命令对应的超时，单位秒。优先级规则是 单个cmd配置 > 类下所有cmd配置 > 全局配置
	 */
	public int getCmdMetaTimeout(CmdMeta cmd) {
		int defaultTimeout = this.globalTimeout;
		return defaultTimeout;
	}

	/**
	 * 初始化Url映射
	 * 
	 * @return
	 */
	public Map<String, CmdMeta> initAutoMap() throws Exception{
		Map<String, CmdMeta> tmp_auto = new LinkedHashMap<String, CmdMeta>();
		String cmdDefaultMethod = config.getCmdDefaultMethod();
		for (BaseCmd cmd : getCmds()) {
			String cmdName = getCmdName(cmd);
			Class<?> clazz = cmd.getClass();
			Method[] mehods = clazz.getDeclaredMethods();
			for (Method method : mehods) {
				if (!isCmdMethod(cmd, method)) {
					continue;
				}
				String methodName = method.getName();
				CmdMapper cmdMapperForMethod = method.getAnnotation(CmdMapper.class);
				CmdAdmin cmdAdminForMethod = method.getAnnotation(CmdAdmin.class);
				// CmdMapper与CmdAdmin同时为空时，说明不需要暴露接口
				if (!methodName.equals(cmdDefaultMethod) && cmdMapperForMethod == null && cmdAdminForMethod == null)
					continue;
				CmdMeta meta = newCmdMeta(cmd, method);
				String methodUrl = MessageFormat.format("/{0}/{1}", cmdName, methodName);
				if (methodName.equals(cmdDefaultMethod))// 默认方法无需方法名
					methodUrl = MessageFormat.format("/{0}", cmdName);
				if (cmdMapperForMethod != null && cmdMapperForMethod.value() != null && cmdMapperForMethod.value().length > 0) {
					String url = cmdMapperForMethod.value()[0];// 只取第一个地址
					if (StringUtils.isNotBlank(url))
						methodUrl = CmdMappers.sanitizePath(url);
				}
				if (tmp_auto.containsKey(methodUrl))
					throw new Exception("Duplicate address Url:" + methodUrl);
				tmp_auto.put(methodUrl, meta);
			}
		}
		cmdAllMap.putAll(tmp_auto);
		log.debug("Netty服务.Url映射.{}", tmp_auto);
		return tmp_auto;
	}

	/**
	 * 是否暴露CMD Http接口
	 * 
	 * @param cmd
	 * @param method
	 * @return
	 */
	private boolean isCmdMethod(BaseCmd cmd, Method method) {
		if (method.getAnnotation(Deprecated.class) != null) {
			return false;
		}
		if (!Modifier.isPublic(method.getModifiers())) {
			return false;
		}
		Class<?>[] pts = method.getParameterTypes();
		if (pts.length != 2) {
			return false;
		}
		if (!pts[0].isAssignableFrom(XLHttpRequest.class) || !pts[1].isAssignableFrom(XLHttpResponse.class)) {
			return false;
		}
		return true;
	}

	/**
	 * <pre>
	 * 2011-01-25 为了解决统计cmd各自的请求量问题,内存中需要统一CmdMeta
	 * 即：如果cmd,method一样,内存中只能用相同的引用
	 */
	private CmdMeta newCmdMeta(BaseCmd cmd, Method method) {
		CmdMeta tmp = new CmdMeta(cmd, method);
		CmdMeta ori = cmdMetaUnite.get(tmp);
		if (ori == null) {
			cmdMetaUnite.put(tmp, tmp);
			return tmp;
		}
		return ori;
	}

	/**
	 * 重置CMD配置,后台扫描器，超时功能是否启动
	 * 
	 * @return
	 */
	public void resetCmdConfig() {
		boolean damonScannerEnable = false;// 默认不启动
		for (CmdMeta meta : cmdMetaUnite.keySet()) {
			meta.timeout = getCmdMetaTimeout(meta);
			if (meta.getTimeout() > 0) {// 只要有个命令有设置超时,后台扫描器就要打开
				damonScannerEnable = true;
			}
		}
		timeoutInterrupter.setThreadInterrupterEnable(damonScannerEnable);
		log.debug("Netty服务.重置CMD配置,后台扫描器，超时功能是否启动.damonScannerEnable=" + damonScannerEnable);
	}
}
