/**
 * 
 */
package com.xunlei.netty.grpcserver.protobuf;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.lang.reflect.Field;
import java.nio.charset.Charset;

import org.apache.commons.lang.StringUtils;

/**
 * 生成Proto文件
 * 
 * @author wangcanyi
 *
 */
public class GenerateProtoFile {
	private String packageName;// 包名
	private String filePath;// 文件存放地址
	private String serviceName;// 服务名
	private boolean isCoverIfExist;// 当文件存在时，是否覆盖
	// /com/xunlei/netty/grpcserver/protobuf/template/ProtoHeader.proto
//	private String protoHeaderFilePath = "template/ProtoHeader.proto";// proto头文件地址
//	private String protoMessageFilePath = "template/ProtoMessage.proto";// proto消息文件地址
//	private String protoMessageFieldFilePath = "template/ProtoMessageField.proto";// proto消息字段文件地址
	private String protoHeaderFilePath = "com/xunlei/netty/grpcserver/protobuf/template/ProtoHeader.proto";// proto头文件地址
	private String protoMessageFilePath = "com/xunlei/netty/grpcserver/protobuf/template/ProtoMessage.proto";// proto消息文件地址
	private String protoMessageFieldFilePath = "com/xunlei/netty/grpcserver/protobuf/template/ProtoMessageField.proto";// proto消息字段文件地址
	private final String protoHeaderTmp;// proto头模板
	private final String protoMessageTmp;// proto消息模板
	private final String protoMessageFieldTmp;// proto消息字段模板

	/**
	 * 构造器
	 * 
	 * @param filePath 文件存放地址
	 * @param packageName 包名
	 * @param serviceName 服务名
	 * @throws Exception
	 */
	public GenerateProtoFile(String filePath, String packageName, String serviceName) throws Exception {
		this(filePath, packageName, serviceName, false);
	}

	/**
	 * 构造器
	 * 
	 * @param filePath 文件存放地址
	 * @param packageName 包名
	 * @param serviceName 服务名
	 * @param isCoverIfExist 当文件存在时，是否覆盖
	 * @throws Exception
	 */
	public GenerateProtoFile(String filePath, String packageName, String serviceName, boolean isCoverIfExist) throws Exception {
		if (StringUtils.isBlank(filePath))
			throw new RuntimeException("filePath为空");
		if (StringUtils.isBlank(packageName))
			throw new RuntimeException("packageName为空");
		if (StringUtils.isBlank(serviceName))
			throw new RuntimeException("serviceName为空");
		this.filePath = filePath;
		this.packageName = packageName;
		this.serviceName = serviceName;
		this.isCoverIfExist = isCoverIfExist;
		this.protoHeaderTmp = this.readFile(this.protoHeaderFilePath);
		this.protoMessageTmp = this.readFile(this.protoMessageFilePath);
		this.protoMessageFieldTmp = this.readFile(this.protoMessageFieldFilePath);
	}

	/**
	 * 生成Proto文件根据DTO反射字段
	 * 
	 * @param classes
	 * @throws Exception
	 */
	public void generateProtoFileByDTO(Class... classes) throws Exception {
		if (classes == null || classes.length == 0)
			throw new RuntimeException("DTO列表为空");
		// 生成proto信息
		StringBuilder protoFileSB = new StringBuilder(this.getProtoHeader());
		protoFileSB.append("\r\n\r\n");
		for (Class classDTO : classes) {
			String classMessage = this.getProtoMessage(classDTO);
			protoFileSB.append(classMessage);
			protoFileSB.append("\r\n\r\n");
		}
		// 写入文件
		this.writeFile(protoFileSB.toString());
	}

	/**
	 * 获取Proto页头
	 * 
	 * @return
	 * @throws Exception
	 */
	private String getProtoHeader() throws Exception {
		String protoHeader = new String(this.protoHeaderTmp);
		if (StringUtils.isNotBlank(protoHeader)) {
			String packageName = this.packageName + ".proto";
			String javaPackageName = this.packageName + ".protobuf";
			String javaClassName = this.serviceName + "Proto";
			protoHeader = protoHeader.replace("{package}", packageName);
			protoHeader = protoHeader.replace("{java_package}", javaPackageName);
			protoHeader = protoHeader.replace("{java_outer_classname}", javaClassName);
		}
		return protoHeader;
	}

	/**
	 * 获取消息信息
	 * 
	 * @param classDTO
	 * @return
	 * @throws Exception
	 */
	private String getProtoMessage(Class classDTO) throws Exception {
		String protoMessage = new String(this.protoMessageTmp);
		StringBuilder protoMessageField = new StringBuilder();
		Field[] fields = classDTO.getDeclaredFields();
		if (fields != null && fields.length > 0) {
			for (int i = 0; i < fields.length; i++) {
				Field field = fields[i];
				String fieldType = this.getProtoFieldType(field.getType());
				String fieldName = field.getName();
				String messageField = new String(this.protoMessageFieldTmp);
				messageField = messageField.replace("{FieldType}", fieldType);
				messageField = messageField.replace("{FieldName}", fieldName);
				messageField = messageField.replace("{FieldIndex}", (i + 1) + "");
				if (i != 0)
					protoMessageField.append("\r\n");
				protoMessageField.append("\t");//添加制表符
				protoMessageField.append(messageField);
			}
		}
		protoMessage = protoMessage.replace("{MessageName}", classDTO.getSimpleName());
		protoMessage = protoMessage.replace("{MessageField}", protoMessageField.toString());
		return protoMessage;
	}

	/**
	 * 获取字段类型
	 * 
	 * @param type
	 * @return
	 */
	private String getProtoFieldType(Class<?> type) {
		if (type != null) {
			if (type == double.class || type == Double.class)
				return "double";
			if (type == float.class || type == Float.class)
				return "float";
			if (type == int.class || type == Integer.class)
				return "int32";
			if (type == long.class || type == Long.class)
				return "int64";
			if (type == boolean.class || type == Boolean.class)
				return "bool";
			if (type == String.class)
				return "string";
		}
		return "UnKnown";
	}

	/**
	 * 文件读取
	 * 
	 * @param filePath 文件地址
	 * @return
	 * @throws Exception
	 */
	private String readFile(String filePath) throws Exception {
		InputStream fis = GenerateProtoFile.class.getClassLoader().getResourceAsStream(filePath);// 当前文件所在文件流
		byte[] buf = new byte[1024];
		StringBuffer sb = new StringBuffer();
		while ((fis.read(buf)) != -1) {
			sb.append(new String(buf,"UTF8"));
			buf = new byte[1024];// 重新生成，避免和上次读取的数据重复
		}
		fis.close();
		return sb.toString().trim();
	}

	private void writeFile(String content) throws Exception {
		File file = new File(this.filePath);
		if (file.isDirectory())
			throw new FileNotFoundException();
		if (!file.exists()) {
			file.createNewFile();
		} else {
			if (!this.isCoverIfExist)
				throw new RuntimeException("文件已存在，请确认是否替换");
		}
		FileOutputStream out = new FileOutputStream(file, false);
		BufferedWriter rd = new BufferedWriter(new OutputStreamWriter(out, "utf-8"));
		// PrintStream ps = new PrintStream(out,false,"UTF-8");
		rd.write(content);
		rd.close();
		out.close();
		//ps.print(content);
		//ps.close();
	}
}
