package com.xunlei.netty.grpcserver.cmd;

import java.util.Map;

import org.jboss.netty.util.internal.ConcurrentHashMap;
import org.slf4j.Logger;

import com.dyuproject.protostuff.LinkedBuffer;
import com.dyuproject.protostuff.ProtostuffIOUtil;
import com.dyuproject.protostuff.Schema;
import com.dyuproject.protostuff.runtime.RuntimeSchema;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.protobuf.ExtensionRegistry;
import com.googlecode.protobuf.format.JsonFormat;
import com.xunlei.common.util.XLRuntimeException;
import com.xunlei.netty.exception.BusinessRuntimeException;
import com.xunlei.netty.httpserver.cmd.BaseCmd;
import com.xunlei.netty.httpserver.model.CookieInfo;
import com.xunlei.netty.soaserver.cmd.BaseSOACmd;
import com.xunlei.util.Log;

/**
 * GRPC接口辅助基类
 * 
 * @author wangcanyi
 *
 */
public class BaseGRPCCmd extends BaseSOACmd {
	private static Logger logger = Log.getLogger(BaseGRPCCmd.class.getName());
	public static final BaseGRPCCmd INSTANCE = new BaseGRPCCmd();
	private JsonFormat protoJson = new JsonFormat();
	private Gson gson = new Gson();
	private Map<Class<?>,Schema<?>> protostuffSchema = new ConcurrentHashMap<Class<?>, Schema<?>>();

	/**
	 * HTTP接口返回 转 GRPC Builder
	 * 
	 * @param builder GRPC Builder实体
	 * @param httpCmd HttpCmd实体
	 * @param methodName HttpCmd方法名
	 * @param param Http Url参数
	 * @throws Exception
	 */
	public void mergeBuilderByHTTP(com.google.protobuf.GeneratedMessageV3.Builder<?> builder, BaseCmd httpCmd, String methodName, Map<String, String> param)
			throws Exception {
		mergeBuilderByHTTP(builder, httpCmd, methodName, param, null);
	}

	/**
	 * HTTP接口返回 转 GRPC Builder
	 * 
	 * @param builder GRPC Builder实体
	 * @param httpCmd HttpCmd实体
	 * @param methodName HttpCmd方法名
	 * @param param Http Url参数
	 * @param cookieInfo cookie信息
	 * @throws Exception
	 */
	public void mergeBuilderByHTTP(com.google.protobuf.GeneratedMessageV3.Builder<?> builder, BaseCmd httpCmd, String methodName, Map<String, String> param,
			CookieInfo cookieInfo) throws Exception {
		JsonObject jsonObject = this.getHTTPMethodJsonObjectWithCookie(httpCmd, methodName, param, cookieInfo);
		if (jsonObject != null) {
			int code = jsonObject.get("rtn").getAsInt();
			if (code == 0) {
				String str = jsonObject.get("data").toString();
				protoJson.merge(str, ExtensionRegistry.getEmptyRegistry(), builder);
			} else {
				String msg = jsonObject.get("data").getAsString();
				throw new BusinessRuntimeException(code + "", msg);
			}
		} else {
			throw new XLRuntimeException("jsonObject为空");
		}
	}

	/**
	 * HTTP接口返回JsonObject
	 * 
	 * @param httpCmd HttpCmd实体
	 * @param methodName HttpCmd方法名
	 * @param param Http Url参数
	 * @param cookieInfo cookie信息
	 * @return Json对象
	 * @throws Exception
	 */
	public JsonObject getHTTPMethodJsonObjectWithCookie(BaseCmd httpCmd, String methodName, Map<String, String> param, CookieInfo cookieInfo) throws Exception {
		JsonObject jsonObject = null;
		Object resultMsg = this.getHTTPMethodResultWithCookie(httpCmd, methodName, param, cookieInfo);
		if (resultMsg != null) {
			// 处理JSONP
			String resultMsgStr = resultMsg.toString().trim();
			if (!resultMsgStr.startsWith("{")) {
				resultMsgStr = resultMsgStr.substring(resultMsgStr.indexOf("{"), resultMsgStr.length() - 1);
			}
			jsonObject = new JsonParser().parse(resultMsgStr).getAsJsonObject();
		}
		return jsonObject;
	}
	
	/**
	 * 对象序列化为 Protobuf byte数组
	 * @param obj Javabean 对象
	 * @return
	 */
	public <T> byte[] serializeProtobuf(T obj) {
		Class<T> cls = (Class<T>) obj.getClass();
		LinkedBuffer buffer = LinkedBuffer.allocate(LinkedBuffer.DEFAULT_BUFFER_SIZE);
		try {
			Schema<T> schema = getProtostuffSchema(cls);
			return ProtostuffIOUtil.toByteArray(obj, schema, buffer);
		} catch (Exception e) {
			throw new IllegalStateException(e.getMessage(), e);
		} finally {
			buffer.clear();
		}
	}
	
//	/**
//	 * 对象反序列化 Javabean 当前序列化顺序错乱，故弃用 
//	 * 
//	 * @param data Protobuf byte数组
//	 * @param cls Javabean Class
//	 * @return
//	 */
//	public <T> T deserializeProtobuf(byte[] data, Class<T> cls) {
//		try {
//			T message = cls.newInstance();
//			Schema<T> schema = getProtostuffSchema(cls);
//			ProtostuffIOUtil.mergeFrom(data, message, schema);
//			return message;
//		} catch (Exception e) {
//			throw new IllegalStateException(e.getMessage(), e);
//		}
//	}
	
	/**
	 * 对象反序列化 Javabean
	 * @param request Protobuf 请求
	 * @param cls Javabean Class
	 * @return
	 */
	public <T> T deserializeProtobuf(com.google.protobuf.GeneratedMessageV3 request, Class<T> cls) {
		return gson.fromJson(protoJson.printToString(request), cls);
	}
	
	/**
	 * 获取Protostuff Schema 
	 * @param cls Javabean 对象的Class
	 * @return
	 */
	private <T> Schema<T> getProtostuffSchema(Class<T> cls){
		Schema<T> schema = (Schema<T>)protostuffSchema.get(cls);
		if(schema == null){
			schema = RuntimeSchema.createFrom(cls);
			if(schema != null)
				protostuffSchema.put(cls, schema);
		}
		return schema;
	}
}
