/**
 * 
 */
package com.xunlei.netty.consul.loadbalance;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;

import com.dianping.cat.Cat;
import com.dianping.cat.message.Transaction;
import com.ecwid.consul.v1.health.model.HealthService;
import com.xunlei.netty.consul.ConsulFactory;
import com.xunlei.netty.util.Log;

/**
 * 轮询法 负载均衡
 * 
 * @author wangcanyi
 *
 */
public class RoundRobinLoadBalance implements ILoadBalance {
	private static final Logger log = Log.getLogger();
	private static final ConcurrentHashMap<String, Integer> mapPos = new ConcurrentHashMap<String, Integer>();

	@Override
	public HealthService chooseByList(String serviceName, List<HealthService> serviceList) {
		HealthService service = null;
		if (StringUtils.isNotBlank(serviceName) && serviceList != null && serviceList.size() > 0) {
			Transaction t = Cat.newTransaction("Consul", "RoundRobinLoadBalance.chooseByList");
			t.addData("serviceName", serviceName);
			t.addData("serviceList", serviceList);
			try {
				List<HealthService> newServiceList = new ArrayList<HealthService>();
				newServiceList.addAll(serviceList);
				synchronized (("LB_LockKey_" + serviceName).intern()) {// 将锁Key放入常量池
					Integer pos = 0;
					if (mapPos.containsKey(serviceName))
						pos = mapPos.get(serviceName);
					if (pos >= newServiceList.size())
						pos = 0;
					service = newServiceList.get(pos);
					pos++;
					mapPos.put(serviceName, pos);
				}

				t.setStatus(Transaction.SUCCESS);
			} catch (Exception e) {
				t.setStatus(e);
				log.error("Consul 选择一个服务[RoundRobinLoadBalance.chooseByList] 异常：" + e.getMessage(), e);
			} finally {
				t.complete();
			}
		}
		return service;
	}

	@Override
	public HealthService choose(String serviceName) {
		HealthService service = null;
		if (StringUtils.isNotBlank(serviceName)) {
			Transaction t = Cat.newTransaction("Consul", "RoundRobinLoadBalance.choose");
			t.addData("serviceName", serviceName);
			try {
				List<HealthService> serviceList = ConsulFactory.getInstance().getServiceList(serviceName);
				service = this.chooseByList(serviceName, serviceList);
				if(service != null){
					Cat.logEvent("Consul.ChooseService.serviceId", service.getService().getId());
					Cat.logEvent("Consul.ChooseService.serviceName", service.getService().getService());
					Cat.logEvent("Consul.ChooseService.serviceAddress", service.getService().getAddress());
					Cat.logEvent("Consul.ChooseService.servicePort", service.getService().getPort() + "");
				}
				t.setStatus(Transaction.SUCCESS);
			} catch (Exception e) {
				t.setStatus(e);
				log.error("Consul 选择一个服务[RoundRobinLoadBalance.choose] 异常：" + e.getMessage(), e);
			} finally {
				t.complete();
			}
		}
		return service;
	}
}
