package com.xunlei.util.hbase;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.*;
import org.apache.hadoop.hbase.client.*;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;

/**
 * Created by yunwang on 2015/12/12.
 */
public class HBase {

    private static Logger log = Logger.getLogger(HBase.class);

    private String    zkHosts;
    private String    zkPort;
    private TableName tableName;

    private Configuration configuration;
    private Connection    connection;
    private Admin         admin;
    private Table         table;
    private ScheduledExecutorService ses = new ScheduledThreadPoolExecutor(1);

    //初始化链接
    public HBase(String zkHosts, String zkPort) throws Exception {
        this.zkHosts = zkHosts;
        this.zkPort = zkPort;

        configuration = HBaseConfiguration.create();
        configuration.set("hbase.zookeeper.quorum", zkHosts);
        configuration.set("hbase.zookeeper.property.clientPort", zkPort);
        configuration.set("zookeeper.znode.parent", "/hbase");
        configuration.set("hbase.rpc.timeout", "10000");
        configuration.set("ipc.socket.timeout", "10000");
        configuration.set("hbase.client.retries.number", "3");
        configuration.set("hbase.client.pause", "1000");
        configuration.set("zookeeper.session.timeout", "10000");
        configuration.set("zookeeper.recovery.retry", "3");
        configuration.set("zookeeper.recovery.retry.intervalmill", "1000");

        connection = ConnectionFactory.createConnection(configuration);
        admin = connection.getAdmin();

        //ses.scheduleWithFixedDelay(check(), 5, 5, TimeUnit.SECONDS);

    }

    public void set(String tableName) throws IOException {
        this.tableName = TableName.valueOf(tableName);
        table = connection.getTable(this.tableName);
    }

    public Runnable check() {

        return new Runnable() {
            public void run() {

                System.out.println("connection checked");
                try {

                    System.out.println(connection.isClosed());
                    System.out.println(connection.isAborted());
                    if (connection.isClosed()) {
                        System.out.println("connection will rebuild");
                        connection = ConnectionFactory.createConnection(configuration);
                        admin = connection.getAdmin();
                        table = connection.getTable(tableName);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    //log.error(e);
                }
            }
        };
    }

    //关闭连接
    public void close() {

        try {
            if (null != table)
                table.close();
        } catch (IOException e) {
            log.warn(e.getMessage(), e);
        }
        try {
            if (null != admin)
                admin.close();
        } catch (IOException e) {
            log.warn(e.getMessage(), e);
        }
        try {
            if (null != connection)
                connection.close();
        } catch (IOException e) {
            log.warn(e.getMessage(), e);
        }

    }

    //建表
    public void createTable(String[] cols) throws IOException {
        if (admin.tableExists(tableName)) {
            log.warn(tableName + " table is exists!");
        } else {
            HTableDescriptor hTableDescriptor = new HTableDescriptor(tableName);
            for (String col : cols) {
                HColumnDescriptor hColumnDescriptor = new HColumnDescriptor(col);
                //hColumnDescriptor.setMaxVersions(100);
                hTableDescriptor.addFamily(hColumnDescriptor);
            }
            admin.createTable(hTableDescriptor);
        }
    }

    //删表
    public void deleteTable() throws IOException {
        if (admin.tableExists(tableName)) {
            admin.disableTable(tableName);
            admin.deleteTable(tableName);
        }
    }

    //查看已有表
    public void listTables() throws IOException {

        HTableDescriptor hTableDescriptors[] = admin.listTables();
        for (HTableDescriptor hTableDescriptor : hTableDescriptors) {
            System.out.println(hTableDescriptor.getNameAsString());
        }
    }

    //插入数据
    public void insertRow(String rowkey, String colFamily, String col, String val)
            throws IOException {
        table.put(row2put(rowkey, colFamily, col, val));

    }

    public void insertRows(List<Put> puts) throws IOException {
        table.put(puts);
    }

    public Put row2put(String rowkey, String colFamily, String col, String val) throws IOException {
        Put put = new Put(Bytes.toBytes(rowkey));
        put.addColumn(Bytes.toBytes(colFamily), Bytes.toBytes(col), Bytes.toBytes(val));
        return put;
    }

    public void putAddColumn(Put put, String colFamily, String col, String val) {
        put.addColumn(Bytes.toBytes(colFamily), Bytes.toBytes(col), Bytes.toBytes(val));

    }

    //删除数据
    public void deleRow(String rowkey, String colFamily, String col) throws IOException {
        Delete delete = new Delete(Bytes.toBytes(rowkey));
        table.delete(delete);
    }

    //删除指定列族
    public void deleColumnFamily(String rowkey, String colFamily, String col) throws IOException {

        Delete delete = new Delete(Bytes.toBytes(rowkey));
        delete.addFamily(Bytes.toBytes(colFamily));
        table.delete(delete);

    }

    //删除指定列
    public void deleColumn(String rowkey, String colFamily, String col) throws IOException {

        Delete delete = new Delete(Bytes.toBytes(rowkey));
        delete.addColumn(Bytes.toBytes(colFamily), Bytes.toBytes(col));
        table.delete(delete);
    }

    //删除多行
    public void deleRows(List<String> rowkeys) throws IOException {
        List<Delete> deleteList = new ArrayList<Delete>();
        for (String rowkey : rowkeys) {
            Delete delete = new Delete(Bytes.toBytes(rowkey));
            deleteList.add(delete);
        }
        table.delete(deleteList);
    }

    //根据rowkey查找数据
    public Result getRowResult(String rowkey) throws IOException {
        Get get = new Get(Bytes.toBytes(rowkey));
        return table.get(get);

    }

    public Result getRowResult(String rowkey, String colFamily) throws IOException {
        Get get = new Get(Bytes.toBytes(rowkey));
        //获取指定列族数据
        get.addFamily(Bytes.toBytes(colFamily));
        return table.get(get);

    }

    public Result getRowResult(String rowkey, String colFamily, String col) throws IOException {
        Get get = new Get(Bytes.toBytes(rowkey));
        //获取指定列族数据
        get.addFamily(Bytes.toBytes(colFamily));
        //获取指定列数据
        get.addColumn(Bytes.toBytes(colFamily), Bytes.toBytes(col));
        return table.get(get);

    }

    public List<Map<String, String>> getRowList(String rowkey) throws IOException {
        return result2List(getRowResult(rowkey));
    }

    public List<Map<String, String>> getRowList(String rowkey, String colFamily)
            throws IOException {
        return result2List(getRowResult(rowkey, colFamily));
    }

    public List<Map<String, String>> getRowList(String rowkey, String colFamily, String col)
            throws IOException {
        return result2List(getRowResult(rowkey, colFamily, col));
    }

    //格式化输出
    public List<Map<String, String>> result2List(Result result) {
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();

        for (Cell cell : result.rawCells()) {
            Map<String, String> map = new HashMap<String, String>();
            map.put("Row", new String(CellUtil.cloneRow(cell)));
            map.put("ColumnFamily", new String(CellUtil.cloneFamily(cell)));
            map.put("Timestamp", String.valueOf(cell.getTimestamp()));
            map.put("Qualifier", new String(CellUtil.cloneQualifier(cell)));
            map.put("StringValue",  new String(CellUtil.cloneValue(cell)));
            list.add(map);
        }

        return list;
    }
//    //格式化输出
//    public List<Map<String, String>> result2List(Result result) {
//        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
//
//        for (Cell cell : result.rawCells()) {
//            Map<String, String> map = new HashMap<String, String>();
//            map.put("RowName", new String(CellUtil.cloneRow(cell)));
//            map.put("ColumnFamily", new String(CellUtil.cloneFamily(cell)));
//            map.put("Timestamp", String.valueOf(cell.getTimestamp()));
//            map.put("Qualifier", new String(CellUtil.cloneQualifier(cell)));
//            map.put("Value", new String(CellUtil.cloneValue(cell)));
//            list.add(map);
//        }
//
//        return list;
//    }

    //批量查找数据

    public List<List<Map<String, String>>> scanData() throws IOException {
        Scan scan = new Scan();
        ResultScanner resultScanner = table.getScanner(scan);

        List<List<Map<String, String>>> list = new ArrayList<List<Map<String, String>>>();
        for (Result result : resultScanner) {
            list.add(result2List(result));
        }

        return list;
    }

    public List<List<Map<String, String>>> scanData(String startRow, String stopRow)
            throws IOException {
        return scanData(Bytes.toBytes(startRow), Bytes.toBytes(stopRow));
    }

    public List<List<Map<String, String>>> scanData(byte[] startRow, byte[] stopRow)
            throws IOException {
        Scan scan = new Scan();
        scan.setStartRow(startRow);
        scan.setStopRow(stopRow);

        ResultScanner resultScanner = table.getScanner(scan);

        List<List<Map<String, String>>> list = new ArrayList<List<Map<String, String>>>();
        for (Result result : resultScanner) {
            list.add(result2List(result));
        }

        return list;
    }

    public List<List<Map<String, String>>> scanData(Scan scan)
            throws IOException {
        ResultScanner resultScanner = table.getScanner(scan);

        List<List<Map<String, String>>> list = new ArrayList<List<Map<String, String>>>();
        for (Result result : resultScanner) {
            list.add(result2List(result));
        }

        return list;
    }

    // getter

    public Configuration getConfiguration() {
        return configuration;
    }

    public Connection getConnection() {
        return connection;
    }

    public Admin getAdmin() {
        return admin;
    }

    public Table getTable() {
        return table;
    }

    public static void main(String[] args) throws Exception {
        //String zkHosts = "twin14125,twin14129,twin14141";
        String zkHosts = "vip19";
        String zkPort = "2181";

        HBase h = new HBase(zkHosts, zkPort);

        //        h.set("t2");
        //        h.createTable(new String[]{"cf1"});

        h.set("wangyunTest");
        //h.insertRow("rw", "f1", "q1", "val");

        for (int i = 0; i < 1000; i++) {

            h.insertRow("rw" + i, "f", "q1", "val" + i);
            System.out.println(i);
            Thread.sleep(3000);

            //            try {
            //                h.insertRow("rw" + i, "f", "q1", "val" + i);
            //                System.out.println(i);
            //                Thread.sleep(3000);
            //            } catch (Exception e) {
            //
            //                System.out.println("err : " + i);
            //            }
        }

        //        h.insertRow("rw1", "cf1", "q1", "val1");
        //        h.getRowResult("rw1", "cf1", "q1");
        //        h.scanData("rw1", "rw2");
        //        h.deleRow("rw1", "cf1", "q1");
        //        h.deleteTable();
    }

}
