package com.xunlei.stream.util.redis.impl;

import com.xunlei.stream.util.redis.RedisApi;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.*;

import java.util.Map;
import java.util.Set;

/**
 * Created by xiongyingqi on 2015/12/15.
 */
public class Redis implements RedisApi {
    private static final Logger logger = LoggerFactory.getLogger(Redis.class);

    private JedisPool       jedisPool;
    private JedisPoolConfig config;

    public Redis() {
    }

    public Redis(String host, int port) {

        initConfig();
        jedisPool = new JedisPool(config, host, port, Protocol.DEFAULT_TIMEOUT);

    }

    public Redis(String host, int port, String pwd) {

        initConfig();
        jedisPool = new JedisPool(config, host, port, Protocol.DEFAULT_TIMEOUT, pwd);

    }

    public void initConfig() {

        config = new JedisPoolConfig();
        //控制一个pool可分配多少个jedis实例，通过pool.getResource()来获取；
        //如果赋值为-1，则表示不限制；如果pool已经分配了maxActive个jedis实例，则此时pool的状态为exhausted(耗尽)。
        config.setMaxTotal(50);
        //控制一个pool最多有多少个状态为idle(空闲的)的jedis实例。
        config.setMaxIdle(5);
        //表示当borrow(引入)一个jedis实例时，最大的等待时间，如果超过等待时间，则直接抛出JedisConnectionException；
        config.setMaxWaitMillis(1000 * 100);
        //在borrow一个jedis实例时，是否提前进行validate操作；如果为true，则得到的jedis实例均是可用的；
        config.setTestOnBorrow(true);

    }

    @Override
    public Jedis getJedis() {
        return jedisPool.getResource();
    }

    @Override
    public void closeJedis(Jedis jedis) {
        if (jedis == null) {
            return;
        }
        jedis.close();
    }

    /**
     * 回收资源
     */
    @Override
    public void returnResource() {

    }

    public String get(String key) {

        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();

            String v = jedis.get(key);
            return v;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    public Map<String, String> hGetAll(String key) {
        Jedis jedis = null;

        try {

            jedis = jedisPool.getResource();
            Map<String, String> mapRes = jedis.hgetAll(key);
            if (logger.isDebugEnabled()) {
                logger.debug("hGetAll... key: {}, result: {}", key, mapRes);
            }
            return mapRes;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    public String hGet(String key, String field) {
        Jedis jedis = null;

        try {

            jedis = jedisPool.getResource();

            String value = jedis.hget(key, field);

            return value;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    public void hSet(String key, String subKey, String value) {

        Jedis jedis = null;

        try {

            jedis = jedisPool.getResource();

            Long hset = jedis.hset(key, subKey, value);

        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
    }

    @Override
    public void hDel(String key, String subKey) {
        Jedis jedis = null;

        try {

            jedis = jedisPool.getResource();
            Long hdel = jedis.hdel(key, subKey);

        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
    }

    public void hDel(String key, String[] subKey) {
        Jedis jedis = null;

        try {

            jedis = jedisPool.getResource();
            Long hdel = jedis.hdel(key, subKey);

        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
    }

    @Override
    public Long sAdd(String key, String value) {
        Jedis jedis = null;

        try {

            jedis = jedisPool.getResource();
            Long rs = jedis.sadd(key, value);
            return rs;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    @Override
    public void zAdd(String key, String member, Integer value) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            jedis.zadd(key, value, member);
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
    }

    public void set(String key, String value, int holdTime) {
        Jedis jedis = null;

        try {

            jedis = jedisPool.getResource();
            String setex = jedis.setex(key, holdTime, value);

        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
    }

    /**
     * 有序集添加操作
     *
     * @param key
     * @param value
     * @param score
     */
    @Override
    public void zadd(String key, double score, String value) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            jedis.zadd(key, score, value);
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
    }

    @Override
    public Set<String> zrange(String key, long start, long end) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            Set<String> zrange = jedis.zrange(key, start, end);
            return zrange;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    @Override
    public Set<Tuple> zrangeWithScores(String key, long start, long end) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            Set<Tuple> zrange = jedis.zrangeWithScores(key, start, end);
            return zrange;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    @Override
    public Set<String> zrangeByScore(String key, double min, double max) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            Set<String> zrangeByScore = jedis.zrangeByScore(key, min, max);
            return zrangeByScore;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    @Override
    public Set<Tuple> zrangeByScoreAndResultWithScores(String key, double min, double max) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            Set<Tuple> zrange = jedis.zrangeByScoreWithScores(key, min, max);
            return zrange;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    @Override
    public Set<String> zrevrangeByScore(String key, double max, double min) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            Set<String> zrevrange = jedis.zrevrangeByScore(key, max, min);
            return zrevrange;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    @Override
    public Set<String> zrevrangeByScore(String key, double max, double min, int offset, int limit) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            Set<String> zrevrange = jedis.zrevrangeByScore(key, max, min, offset, limit);
            return zrevrange;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    @Override
    public Set<String> zrevrange(String key, long start, long end) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            Set<String> zrevrange = jedis.zrevrange(key, start, end);
            return zrevrange;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    @Override
    public Set<Tuple> zrevrangeWithScores(String key, long start, long end) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            Set<Tuple> zrevrange = jedis.zrevrangeWithScores(key, start, end);
            return zrevrange;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    @Override
    public Set<Tuple> zrevrangeByScoreAndResultWithScores(String key, double max, double min) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            Set<Tuple> zrevrange = jedis.zrevrangeByScoreWithScores(key, max, min);
            return zrevrange;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    public String set(String key, String value) {
        Jedis jedis = null;

        try {
            jedis = jedisPool.getResource();
            String setex = jedis.set(key, value);
            return setex;
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (jedis != null)
                jedis.close();
        }
        return null;
    }

    public JedisPoolConfig getConfig() {
        return config;
    }

    public JedisPool getJedisPool() {
        return jedisPool;
    }

    public void setJedisPool(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
    }

    public void setConfig(JedisPoolConfig config) {
        this.config = config;
    }
    
    public static void main(String[] args){
        Redis redis = new Redis("127.0.0.1", 6379, "test");
        redis.hSet("a", "b", "c");
        redis.hSet("a", "c", "d");
        Map<String, String> map = redis.hGetAll("a");
        System.out.println(map);
        redis.hDel("a", new String[]{"b", "c"});
    }
}
