package com.xunlei.stream.util.redis;

import com.xunlei.stream.factory.Factory;
import com.xunlei.stream.util.redis.impl.RedisClientProvider;
import com.xunlei.common.ClassHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.ResourceBundle;

/**
 * 负责生产redis对象，这样做的目的是更好的测试<br>
 * 如果要模拟redis，则可以实现{@link RedisApi}接口和{@link RedisProvider}接口
 * @author <a href="http://xiongyingqi.com">qi</a>
 * @version 2015-10-22 17:05
 */
public abstract class RedisFactory implements Serializable, Factory {
    private static final Logger logger = LoggerFactory.getLogger(RedisFactory.class);
    private static RedisProvider redisProvider;
    public static final RedisProvider DEFAULT_REDIS_PROVIDER = new RedisClientProvider();

    static {
        initProvider();
    }

    @SuppressWarnings("unchecked")
    public static void initProvider() {
        ResourceBundle bundle = ResourceBundle.getBundle(PROFILE_BUNDLE_NAME);
        String providerClassStr = bundle.getString("RedisFactory.provider");
        try {
            Class<?> providerClass = Class.forName(providerClassStr);
            boolean implementsInterface = ClassHelper
                    .isImplementsInterface(providerClass, RedisProvider.class);
            logger.info("init profile: {}, provider class: {}, implements IpAreaProvider: {}",
                    PROFILE_BUNDLE_NAME, providerClass, implementsInterface);
            if (!implementsInterface) {
                return;
            }
            Class<RedisProvider> redisProviderClass = (Class<RedisProvider>) providerClass;
            RedisProvider redisProvider = redisProviderClass.newInstance();
            setRedisProvider(redisProvider);
        } catch (ClassNotFoundException e) {
            logger.error("load profile: " + PROFILE_BUNDLE_NAME + " and class: " + providerClassStr
                    + " not found! with exception: " + e.getMessage(), e);
        } catch (InstantiationException e) {
            logger.error("class: " + providerClassStr
                    + " instantiation error with message: " + e.getMessage(), e);
        } catch (IllegalAccessException e) {
            logger.error("load profile: " + PROFILE_BUNDLE_NAME + " and class: " + providerClassStr
                    + " caught IllegalAccessException! message: " + e.getMessage(), e);
        }
    }

    public static RedisApi newRedisApi(String instanceName) throws RedisProviderNotSetException {
        if (redisProvider == null) {
//            throw new RedisProviderNotSetException("please set the redisProvider first!");
            redisProvider = DEFAULT_REDIS_PROVIDER;
        }
        return redisProvider.newRedisClient(instanceName);
    }

    public static void setRedisProvider(RedisProvider redisProvider) {
        if (RedisFactory.redisProvider != null) {
            logger.warn("setRedisProvider... reset provider: old provider = {}, new provider = {}", RedisFactory.redisProvider, redisProvider);
        } else {
            logger.info("setRedisProvider... set provider: {}", redisProvider);
        }
        RedisFactory.redisProvider = redisProvider;
    }

    public static RedisProvider getRedisProvider() {
        return redisProvider;
    }
}
