package com.xunlei.stream.util

import java.util.ArrayList
import java.util.List

import org.apache.hadoop.hbase.CellUtil
import org.apache.hadoop.hbase.client.Result
import org.apache.hadoop.hbase.util.Bytes

import scala.collection.mutable

/**
 * @author  <a href="http://xiongyingqi.com">qi</a>
 * @version 2015-08-20 21:05
 */
trait HBaseHelper {

  def bytesToBoolean = (x: Array[Byte]) => Bytes.toBoolean(x)

  def bytesToString = (x: Array[Byte]) => Bytes.toString(x)

  def bytesToShort = (x: Array[Byte]) => Bytes.toShort(x)

  def bytesToInt = (x: Array[Byte]) => Bytes.toInt(x)

  def bytesToLong = (x: Array[Byte]) => Bytes.toLong(x)

  def bytesToDouble = (x: Array[Byte]) => Bytes.toDouble(x)

  def bytesToBigDecimal = (x: Array[Byte]) => Bytes.toBigDecimal(x)

  def bytesToByteArrays = (x: Array[Byte]) => Bytes.toByteArrays(x)

  /**
   * 打印HBase的数据
   * <pre>
   * printHBaseData(scanList, ("name", (x: Array[Byte]) => Bytes.toString(x)), ("age", (x: Array[Byte]) => Bytes.toInt(x)), ("address", (x: Array[Byte]) => Bytes.toString(x)))
   * </pre>
   * @param list 数据
   * @param columns 列表，必须带转换方法，可以使用bytesToBoolean、bytesToString、bytesToShort、bytesToInt、bytesToLong、bytesToDouble、bytesToBigDecimal、bytesToByteArrays
   * @see #bytesToBoolean
   * @see #bytesToString
   * @see #bytesToShort
   * @see #bytesToInt
   * @see #bytesToLong
   * @see #bytesToDouble
   * @see #bytesToBigDecimal
   * @see #bytesToByteArrays
   */
  def printHBaseData(list: Array[(Array[Byte], List[(Array[Byte], Array[Byte], Array[Byte])])], idConvert: Array[Byte] => Any, columns: (String, Array[Byte] => Any)*): Unit = {
    list.foreach(f => {
      val result = parseResult(f, idConvert, columns: _*) // :_* 表示逐个传入
      println("--------------------------------------")
      println("id: " + result._1)
      for (i <- 0 to columns.size - 1) {
        val column = columns.apply(i)
        result._2(i) match {
          case Nil => println(column._1 + ": <no such column>")
          case _ => println(column._1 + ": " + result._2(i))
        }
      }
    })
  }

  /**
   * 将result转换为{@link scala.collection.immutable.Map}
   * @param data (Array[Byte], java.util.List[(Array[Byte], Array[Byte], Array[Byte])])
   * @return id -> ([name->value,...])
   */
  def parseResultToMap(data: (Array[Byte], List[(Array[Byte], Array[Byte], Array[Byte])])): (Array[Byte], Map[String, Array[Byte]]) = {
    val map = new mutable.HashMap[String, Array[Byte]]()
    val it = data._2.iterator
    while (it.hasNext) {
      val node = it.next
      val name = new String(node._2)
      val value = node._3
      map.put(name, value)
    }
    (data._1, map.toMap)
  }


  /**
   * 将result转换为(id -> [value1,value2...])
   * @param data (Array[Byte], java.util.List[(Array[Byte], Array[Byte], Array[Byte])])
   * @param idConvert row的转换函数
   * @param columns [列名->转换函数,...]
   * @return (id -> [value1,value2...])顺序和类型跟传入的传入的idConvert和columns对应
   */
  def parseResult(data: (Array[Byte], List[(Array[Byte], Array[Byte], Array[Byte])]), idConvert: Array[Byte] => Any, columns: (String, Array[Byte] => Any)*): (Any, Seq[Any]) = {
    val map = parseResultToMap(data)

    val id = idConvert(data._1)
    val rs = columns.map(
      x => {
        val col = map._2.get(x._1) match {
          case a: Some[Array[Byte]] => {
            val value = x._2.apply(a.get)
            value
          }
          case _ => Nil
        }
        col
      }
    )
    (id, rs)
  }


  /**
   * 将result转换为(id -> [value1,value2...])
   * @param result HBase Result
   * @param idConvert row的转换函数
   * @param columns [列名->转换函数,...]
   * @return (id -> [value1,value2...])顺序和类型跟传入的传入的idConvert和columns对应
   */
  def parseResult(result: Result, idConvert: Array[Byte] => Any, columns: (String, Array[Byte] => Any)*): Option[(Any, Seq[Any])] = {
    val data = parseResult(result) match {
      case None => None
      case Some(s) => s
    }
    data match {
      case x: (Array[Byte], List[(Array[Byte], Array[Byte], Array[Byte])]) => Some(parseResult(x, idConvert, columns: _*)) //:_* 表示逐个传入
      case _ => None
    }
  }

  /**
   * 将result转换为 <b>id -> (family, name, value)</b>
   * @param result HBase Result
   * @return id -> (family, name, value)
   */
  def parseResult(result: Result): Option[(Array[Byte], List[(Array[Byte], Array[Byte], Array[Byte])])] = {
    if (result == null || result == None) {
      return None
    }
    result.isEmpty match {
      case true => return None
      case _ =>
    }
    val row = result.getRow
    val it = result.listCells().iterator()
    val list = new ArrayList[(Array[Byte], Array[Byte], Array[Byte])]()

    while (it.hasNext()) {
      val cell = it.next()
      list.add((CellUtil.cloneFamily(cell), CellUtil.cloneQualifier(cell), CellUtil.cloneValue(cell)))
    }
    Some(row, list)
  }


}
