package com.xunlei.emoticon.mall.web.api.v1;

import com.google.common.base.MoreObjects;
import com.google.common.base.Splitter;
import com.xunlei.emoticon.mall.constant.Action;
import com.xunlei.emoticon.mall.constant.CommonConstant;
import com.xunlei.emoticon.mall.constant.DeviceSize;
import com.xunlei.emoticon.mall.constant.DeviceType;
import com.xunlei.emoticon.mall.constant.Type;
import com.xunlei.emoticon.mall.domain.Configuration;
import com.xunlei.emoticon.mall.domain.Emoticon;
import com.xunlei.emoticon.mall.domain.EmoticonFavorite;
import com.xunlei.emoticon.mall.domain.EmoticonPack;
import com.xunlei.emoticon.mall.domain.EmoticonPackFavorite;
import com.xunlei.emoticon.mall.domain.Feedback;
import com.xunlei.emoticon.mall.domain.Recorder;
import com.xunlei.emoticon.mall.dto.Return;
import com.xunlei.emoticon.mall.service.APIService;
import org.apache.commons.lang3.EnumUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * @author Shunli
 */
@RestController
@RequestMapping("/api/v1/channels/{channelId}")
public class APIController {
    private static final Splitter.MapSplitter HEADER_MAP_SPLITTER = Splitter.on(";").omitEmptyStrings().trimResults().withKeyValueSeparator("=");
    private static final String DEVICE_HEADER = "Device";

    @Autowired
    private APIService apiService;

    private String getDeviceId(String device) {
        return HEADER_MAP_SPLITTER.split(StringUtils.defaultString(device)).get("id");
    }

    private String getDeviceType(String device) {
        String systemInfo = HEADER_MAP_SPLITTER.split(StringUtils.defaultString(device)).get("systemInfo");
        return StringUtils.substringBefore(systemInfo, "/");
    }

    @GetMapping
    public Return load(@PathVariable Long channelId) {
        return CommonConstant.SUCCESS_RETURN.result(apiService.load(channelId));
    }

    @PostMapping("/feedback")
    public Return submitFeedback(@PathVariable Long channelId, @RequestHeader(value = DEVICE_HEADER, required = false) String device,
                                 @RequestParam String content, @RequestParam(required = false) String contact) {
        Feedback feedback = Feedback.builder().channelId(channelId).device(device).content(content).contact(contact).archived(false).build();
        apiService.submitFeedback(feedback);
        return CommonConstant.SUCCESS_RETURN;
    }

    @GetMapping("/emoticon-packs")
    public Return getAllEmoticonPacks(@PathVariable Long channelId,
                                      @PageableDefault(sort = {"priority", "lastModifiedDate"}, direction = Sort.Direction.DESC) Pageable pageable) {
        return CommonConstant.SUCCESS_RETURN.result(apiService.getAllEmoticonPacks(channelId, pageable));
    }

    @GetMapping("/emoticon-packs/{emoticonPackId}/emoticons")
    public Return getOneEmoticonPack(@PathVariable Long channelId, @PathVariable Long emoticonPackId,
                                     @PageableDefault(sort = {"priority", "lastModifiedDate"}, direction = Sort.Direction.DESC) Pageable pageable) {
        return CommonConstant.SUCCESS_RETURN.result(apiService.getOneEmoticonPack(channelId, emoticonPackId, pageable));
    }

    @GetMapping("/emoticon-packs/favorites")
    public Return getFavoriteEmoticonPacks(@PathVariable Long channelId, @RequestHeader(value = DEVICE_HEADER, required = false) String device,
                                           @PageableDefault(sort = {"lastModifiedDate"}, direction = Sort.Direction.DESC) Pageable pageable) {
        return CommonConstant.SUCCESS_RETURN.result(apiService.getFavoriteEmoticonPacks(channelId, getDeviceId(device), pageable));
    }

    @PostMapping("/emoticon-packs/favorites")
    public Return favoriteEmoticonPack(@PathVariable Long channelId, @RequestHeader(value = DEVICE_HEADER, required = false) String device, @RequestParam Long id) {
        EmoticonPack emoticonPack = apiService.findEmoticonPackById(id);
        if (emoticonPack == null) {
            return CommonConstant.FAILED_RETURN.msg("收藏失败");
        }

        String deviceId = getDeviceId(device);
        EmoticonPackFavorite emoticonPackFavorite = apiService.findEmoticonPackFavorite(id, deviceId);
        if (emoticonPackFavorite != null) {
            return CommonConstant.FAILED_RETURN.msg("已经收藏");
        }

        apiService.addEmoticonPackFavorite(EmoticonPackFavorite.builder().deviceId(deviceId).emoticonPack(emoticonPack).build());
        apiService.addRecorder(Recorder.builder().action(Action.FAVORITE).type(Type.EMOTICON_PACK).sourceId(id).deviceId(deviceId).build());

        return CommonConstant.SUCCESS_RETURN.msg("收藏成功");
    }

    @DeleteMapping("/emoticon-packs/favorites")
    public Return unFavoriteEmoticonPack(@PathVariable Long channelId, @RequestHeader(value = DEVICE_HEADER, required = false) String device, @RequestParam Long id) {
        EmoticonPackFavorite emoticonPackFavorite = apiService.findEmoticonPackFavorite(id, getDeviceId(device));
        if (emoticonPackFavorite == null) {
            return CommonConstant.FAILED_RETURN.msg("取消收藏失败");
        }

        apiService.deleteEmoticonPackFavorite(emoticonPackFavorite);
        return CommonConstant.SUCCESS_RETURN.msg("取消收藏成功");
    }

    @PostMapping("/emoticon-packs/shares")
    public Return shareEmoticonPack(@PathVariable Long channelId, @RequestParam Long id, @RequestHeader(value = DEVICE_HEADER, required = false) String device) {
        apiService.addRecorder(Recorder.builder().action(Action.SHARE).type(Type.EMOTICON_PACK).sourceId(id).deviceId(getDeviceId(device)).build());
        return CommonConstant.SUCCESS_RETURN;
    }

    @GetMapping("/emoticons")
    public Return getPopularityEmoticons(@PathVariable Long channelId,
                                         @PageableDefault(sort = {"popularity", "lastModifiedDate"}, direction = Sort.Direction.DESC) Pageable pageable) {
        return CommonConstant.SUCCESS_RETURN.result(apiService.getPopularityEmoticons(channelId, pageable));
    }

    @GetMapping("/emoticons/favorites")
    public Return getFavoriteEmoticons(@PathVariable Long channelId, @RequestHeader(value = DEVICE_HEADER, required = false) String device,
                                       @PageableDefault(sort = {"lastModifiedDate"}, direction = Sort.Direction.DESC) Pageable pageable) {
        return CommonConstant.SUCCESS_RETURN.result(apiService.getFavoriteEmoticons(channelId, getDeviceId(device), pageable));
    }

    @PostMapping("/emoticons/favorites")
    public Return favoriteEmoticon(@PathVariable Long channelId, @RequestHeader(value = DEVICE_HEADER, required = false) String device, @RequestParam Long id) {
        Emoticon emoticon = apiService.findEmoticonById(id);
        if (emoticon == null) {
            return CommonConstant.FAILED_RETURN.msg("收藏失败");
        }

        String deviceId = getDeviceId(device);
        EmoticonFavorite emoticonFavorite = apiService.findEmoticonFavorite(id, deviceId);
        if (emoticonFavorite != null) {
            return CommonConstant.FAILED_RETURN.msg("已经收藏");
        }

        apiService.addEmoticonFavorite(EmoticonFavorite.builder().deviceId(deviceId).emoticon(emoticon).build());
        apiService.addRecorder(Recorder.builder().action(Action.FAVORITE).type(Type.EMOTICON).sourceId(id).deviceId(deviceId).build());

        return CommonConstant.SUCCESS_RETURN.msg("收藏成功");
    }

    @DeleteMapping("/emoticons/favorites")
    public Return unFavoriteEmoticon(@PathVariable Long channelId, @RequestHeader(value = DEVICE_HEADER, required = false) String device, @RequestParam Long id) {
        EmoticonFavorite emoticonFavorite = apiService.findEmoticonFavorite(id, getDeviceId(device));
        if (emoticonFavorite == null) {
            return CommonConstant.FAILED_RETURN.msg("取消收藏失败");
        }

        apiService.deleteEmoticonFavorite(emoticonFavorite);
        return CommonConstant.SUCCESS_RETURN.msg("取消收藏成功");
    }

    @PostMapping("/emoticons/shares")
    public Return shareEmoticon(@PathVariable Long channelId, @RequestParam Long id, @RequestHeader(value = DEVICE_HEADER, required = false) String device) {
        apiService.addRecorder(Recorder.builder().action(Action.SHARE).type(Type.EMOTICON).sourceId(id).deviceId(getDeviceId(device)).build());
        return CommonConstant.SUCCESS_RETURN;
    }

    @PostMapping("/emoticons/downloads")
    public Return downloadEmoticon(@PathVariable Long channelId, @RequestParam Long id, @RequestHeader(value = DEVICE_HEADER, required = false) String device) {
        apiService.addRecorder(Recorder.builder().action(Action.DOWNLOAD).type(Type.EMOTICON).sourceId(id).deviceId(getDeviceId(device)).build());
        return CommonConstant.SUCCESS_RETURN;
    }

    @GetMapping("/ad")
    public Return getAd(@PathVariable Long channelId, @RequestHeader(value = DEVICE_HEADER, required = false) String device, @RequestParam String size) {
        DeviceType deviceType = EnumUtils.getEnum(DeviceType.class, StringUtils.upperCase(getDeviceType(device)));
        DeviceSize deviceSize = EnumUtils.getEnum(DeviceSize.class, StringUtils.upperCase(size));
        return CommonConstant.SUCCESS_RETURN.result(apiService.getEnabledAd(channelId, deviceType, deviceSize));
    }

    @GetMapping("/android-version")
    public Return getAndroidVersion(@PathVariable Long channelId) {
        return CommonConstant.SUCCESS_RETURN.result(apiService.getAndroidUpdater(channelId));
    }

    @GetMapping("/download-status")
    public Return getDownloadStatus(@PathVariable Long channelId) {
        Configuration configuration = apiService.getConfiguration(channelId);
        return CommonConstant.SUCCESS_RETURN.result(MoreObjects.firstNonNull(configuration.getDownloaded(), false));
    }

}
