package com.xunlei.emoticon.mall.service;

import com.google.common.base.MoreObjects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xunlei.emoticon.mall.constant.CommonConstant;
import com.xunlei.emoticon.mall.domain.Channel;
import com.xunlei.emoticon.mall.domain.Emoticon;
import com.xunlei.emoticon.mall.domain.EmoticonPack;
import com.xunlei.emoticon.mall.domain.EmoticonPopularity;
import com.xunlei.emoticon.mall.domain.EmoticonTag;
import com.xunlei.emoticon.mall.repository.ChannelRepository;
import com.xunlei.emoticon.mall.repository.EmoticonPackRepository;
import com.xunlei.emoticon.mall.repository.EmoticonPopularityRepository;
import com.xunlei.emoticon.mall.repository.EmoticonRepository;
import com.xunlei.emoticon.mall.repository.EmoticonTagRepository;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.imageio.ImageIO;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipFile;

/**
 * @author Shunli
 */
@Service
@Slf4j
public class AdminService {
    @Autowired
    private ChannelRepository channelRepository;
    @Autowired
    private EmoticonPackRepository emoticonPackRepository;
    @Autowired
    private EmoticonRepository emoticonRepository;
    @Autowired
    private EmoticonTagRepository emoticonTagRepository;
    @Autowired
    private EmoticonPopularityRepository emoticonPopularityRepository;

    @Autowired
    private IdGenerator idGenerator;

    @Value("${img.path}")
    private String imgPath;
    @Value("${img.url.prefix}")
    private String imgUrlPrefix;
    @Value("${file.upload.path:/tmp}")
    private String fileUploadPath;

    private void changeOwnerToNobody(Path path) {
        try {
            String nobody = "nobody";
            UserPrincipalLookupService lookupService = FileSystems.getDefault().getUserPrincipalLookupService();

            Files.setOwner(path, lookupService.lookupPrincipalByName(nobody));
            Files.getFileAttributeView(path, PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS)
                    .setGroup(lookupService.lookupPrincipalByGroupName(nobody));
        } catch (Exception e) {
            log.error("change owner to nobody failed " + path, e);
        }
    }

    public String saveImg(String fileExtension, InputStream inputStream) throws IOException {
        String fileName = idGenerator.nextId() + "." + fileExtension;

        Path path = Paths.get(imgPath, fileName);
        Files.copy(inputStream, path);
        changeOwnerToNobody(path);

        return imgUrlPrefix + fileName;
    }

    private void pickImages(String fileBaseName, String fileExtension, InputStream inputStream,
                            Map<String, Pair<String, InputStream>> gifMap,
                            Map<String, Pair<String, InputStream>> normalMap) {
        if (StringUtils.equalsIgnoreCase("gif", fileExtension)) {
            gifMap.put(fileBaseName, Pair.of(fileExtension, inputStream));
        } else {
            normalMap.put(fileBaseName, Pair.of(fileExtension, inputStream));
        }
    }

    public String uploadEmoticon(Long channelId, Long emoticonPackId, MultipartFile file) throws IOException {
        Map<String, Pair<String, InputStream>> gifMap = Maps.newHashMap();
        Map<String, Pair<String, InputStream>> normalMap = Maps.newHashMap();

        StringBuilder failedMsg = new StringBuilder();
        String uploadFileName = file.getOriginalFilename();
        String fileExtension = FilenameUtils.getExtension(uploadFileName);

        Path path = Files.write(Paths.get(fileUploadPath, uploadFileName), file.getBytes());
        if (StringUtils.equalsIgnoreCase("zip", fileExtension)) {
            ZipFile zipFile = new ZipFile(path.toFile(), Charset.forName("GBK"));
            zipFile.stream().forEach(zipEntry -> {
                if (!zipEntry.isDirectory()) {
                    String entryName = zipEntry.getName();
                    try {
                        if (!StringUtils.startsWithIgnoreCase(Files.probeContentType(Paths.get(entryName)), "image")) {
                            failedMsg.append("表情“").append(entryName).append("”导入失败，原因：只支持图片");
                        } else {
                            String baseName = FilenameUtils.getBaseName(entryName);
                            String extension = FilenameUtils.getExtension(entryName);
                            pickImages(baseName, extension, zipFile.getInputStream(zipEntry), gifMap, normalMap);
                        }
                    } catch (IOException e) {
                        log.error("upload failed: " + entryName, e);
                        failedMsg.append("表情“").append(entryName).append("”导入失败，原因：").append(e.getMessage()).append("；");
                    }
                }
            });
        } else {
            String name = FilenameUtils.getBaseName(uploadFileName);
            pickImages(name, fileExtension, new FileInputStream(path.toFile()), gifMap, normalMap);
        }

        List<Emoticon> emoticons = Lists.newArrayList();
        for (Map.Entry<String, Pair<String, InputStream>> entry : gifMap.entrySet()) {
            String name = entry.getKey();

            Pair<String, InputStream> normalPair = normalMap.get(name);
            Pair<String, InputStream> gifPair = entry.getValue();

            Emoticon.EmoticonBuilder builder = Emoticon.builder()
                    .channelId(channelId)
                    .emoticonPackId(emoticonPackId)
                    .name(name)
                    .priority(CommonConstant.DEFAULT_PRIORITY)
                    .isGIF(true)
                    .enabled(false);

            if (normalPair != null) {
                builder.displayUrl(saveImg(normalPair.getLeft(), normalPair.getRight()))
                        .downloadUrl(saveImg(gifPair.getLeft(), gifPair.getRight()));
                normalMap.remove(name);
            } else {
                byte[] bytes = IOUtils.toByteArray(gifPair.getRight());
                ByteArrayOutputStream output = new ByteArrayOutputStream();

                ImageIO.write(ImageIO.read(new ByteArrayInputStream(bytes)), "png", output);

                builder.displayUrl(saveImg("png", new ByteArrayInputStream(output.toByteArray())))
                        .downloadUrl(saveImg(gifPair.getLeft(), new ByteArrayInputStream(bytes)));
                failedMsg.append("动态表情“").append(name).append("”没有找到对应的封面图，使用系统生成的封面图；");
            }

            emoticons.add(builder.build());
        }

        for (Map.Entry<String, Pair<String, InputStream>> entry : normalMap.entrySet()) {
            String name = entry.getKey();
            Pair<String, InputStream> pair = entry.getValue();
            String url = saveImg(pair.getLeft(), pair.getRight());

            Emoticon emoticon = Emoticon.builder()
                    .channelId(channelId)
                    .emoticonPackId(emoticonPackId)
                    .name(name)
                    .displayUrl(url)
                    .downloadUrl(url)
                    .isGIF(false)
                    .priority(CommonConstant.DEFAULT_PRIORITY)
                    .enabled(false)
                    .build();

            emoticons.add(emoticon);
        }

        Iterable<Emoticon> saveEmoticons = emoticonRepository.save(emoticons);

        // create Emoticon popularity
        List<EmoticonPopularity> popularityList = Lists.newArrayList();
        saveEmoticons.forEach(emoticon -> popularityList.add(new EmoticonPopularity(emoticon, 0L)));
        emoticonPopularityRepository.save(popularityList);

        // create Emoticon tag
        List<EmoticonTag> tags = Lists.newArrayList();
        Channel channel = channelRepository.findOne(channelId);
        if (channel != null) {
            saveEmoticons.forEach(emoticon -> tags.add(new EmoticonTag(emoticon, channel.getName())));
        }
        EmoticonPack emoticonPack = emoticonPackRepository.findOne(MoreObjects.firstNonNull(emoticonPackId, 0L));
        if (emoticonPack != null) {
            saveEmoticons.forEach(emoticon -> tags.add(new EmoticonTag(emoticon, emoticonPack.getName())));
        }
        emoticonTagRepository.save(tags);

        StringBuffer msg = new StringBuffer();
        msg.append("成功导入").append(Iterables.size(saveEmoticons)).append("个表情。");
        msg.append(failedMsg);

        log.info("upload result {}", msg);
        return msg.toString();
    }
}
