package com.xunlei.emoticon.mall.service;

import com.xunlei.emoticon.mall.constant.CommonConstant;
import com.xunlei.emoticon.mall.constant.DeviceSize;
import com.xunlei.emoticon.mall.constant.DeviceType;
import com.xunlei.emoticon.mall.constant.Type;
import com.xunlei.emoticon.mall.domain.AdImage;
import com.xunlei.emoticon.mall.domain.AndroidUpdater;
import com.xunlei.emoticon.mall.domain.Channel;
import com.xunlei.emoticon.mall.domain.Configuration;
import com.xunlei.emoticon.mall.domain.Emoticon;
import com.xunlei.emoticon.mall.domain.EmoticonFavorite;
import com.xunlei.emoticon.mall.domain.EmoticonPack;
import com.xunlei.emoticon.mall.domain.EmoticonPackFavorite;
import com.xunlei.emoticon.mall.domain.Feedback;
import com.xunlei.emoticon.mall.domain.Recorder;
import com.xunlei.emoticon.mall.domain.SlideShow;
import com.xunlei.emoticon.mall.dto.AdImageDto;
import com.xunlei.emoticon.mall.dto.AndroidUpdaterDto;
import com.xunlei.emoticon.mall.dto.EmoticonDto;
import com.xunlei.emoticon.mall.dto.EmoticonPackDto;
import com.xunlei.emoticon.mall.dto.EmoticonWrapper;
import com.xunlei.emoticon.mall.dto.Landing;
import com.xunlei.emoticon.mall.dto.SlideShowDto;
import com.xunlei.emoticon.mall.repository.AdImageRepository;
import com.xunlei.emoticon.mall.repository.AndroidUpdaterRepository;
import com.xunlei.emoticon.mall.repository.ChannelRepository;
import com.xunlei.emoticon.mall.repository.ConfigurationRepository;
import com.xunlei.emoticon.mall.repository.EmoticonFavoriteRepository;
import com.xunlei.emoticon.mall.repository.EmoticonPackFavoriteRepository;
import com.xunlei.emoticon.mall.repository.EmoticonPackRepository;
import com.xunlei.emoticon.mall.repository.EmoticonPopularityRepository;
import com.xunlei.emoticon.mall.repository.EmoticonRepository;
import com.xunlei.emoticon.mall.repository.FeedbackRepository;
import com.xunlei.emoticon.mall.repository.RecorderRepository;
import com.xunlei.emoticon.mall.repository.SlideShowRepository;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author Shunli
 */
@Service
public class APIService {
    @Autowired
    private ChannelRepository channelRepository;
    @Autowired
    private EmoticonPackRepository emoticonPackRepository;
    @Autowired
    private EmoticonRepository emoticonRepository;
    @Autowired
    private FeedbackRepository feedbackRepository;
    @Autowired
    private RecorderRepository recorderRepository;
    @Autowired
    private SlideShowRepository slideShowRepository;
    @Autowired
    private EmoticonFavoriteRepository emoticonFavoriteRepository;
    @Autowired
    private EmoticonPackFavoriteRepository emoticonPackFavoriteRepository;
    @Autowired
    private EmoticonPopularityRepository emoticonPopularityRepository;
    @Autowired
    private ConfigurationRepository configurationRepository;
    @Autowired
    private AdImageRepository adImageRepository;
    @Autowired
    private AndroidUpdaterRepository androidUpdaterRepository;

    private SlideShowDto buildSlideShowDto(SlideShow source) {
        if (source == null) {
            return null;
        }

        SlideShowDto target = new SlideShowDto();
        BeanUtils.copyProperties(source, target);
        return target;
    }

    private EmoticonPackDto buildEmoticonPackDto(EmoticonPack source) {
        if (source == null) {
            return null;
        }

        EmoticonPackDto target = new EmoticonPackDto();
        BeanUtils.copyProperties(source, target);

        return target;
    }

    private EmoticonDto buildEmoticonDto(Emoticon source) {
        if (source == null) {
            return null;
        }

        EmoticonDto target = new EmoticonDto();
        BeanUtils.copyProperties(source, target);

        return target;
    }

    private AndroidUpdaterDto buildAndroidUpdaterDto(AndroidUpdater source) {
        if (source == null) {
            return new AndroidUpdaterDto();
        }

        AndroidUpdaterDto target = new AndroidUpdaterDto();
        BeanUtils.copyProperties(source, target);

        return target;
    }

    private AdImageDto buildAdImageDto(AdImage source) {
        if (source == null) {
            return null;
        }

        AdImageDto target = new AdImageDto();
        BeanUtils.copyProperties(source, target);

        return target;
    }

    public Landing load(Long channelId) {
        Channel channel = channelRepository.findByIdAndEnabledTrue(channelId);
        if (channel == null) {
            return new Landing();
        }

        Configuration configuration = getConfiguration(channelId);
        int random = configuration.getRandomQty();
        boolean downloaded = configuration.getDownloaded();

        return Landing.builder()
                .name(channel.getName())
                .downloaded(downloaded)
                .slideShows(slideShowRepository.findByChannelIdAndEnabledTrue(channelId, CommonConstant.DEFAULT_SORT)
                        .map(this::buildSlideShowDto)
                        .collect(Collectors.toList()))
                .emoticonWrappers(buildEmoticonWrappers(channelId))
                .emoticons(emoticonRepository.findRandomByChannelIdAndEnabledTrue(channelId, random)
                        .map(this::buildEmoticonDto)
                        .collect(Collectors.toList()))
                .build();
    }

    private List<EmoticonWrapper> buildEmoticonWrappers(Long channelId) {
        return emoticonPackRepository.findByChannelIdAndRecommendedTrueAndEnabledTrue(channelId, CommonConstant.DEFAULT_SORT)
                .map(emoticonPack -> {
                    List<EmoticonDto> emoticons = emoticonRepository
                            .findTop6ByEmoticonPackIdAndEnabledTrue(emoticonPack.getId(), CommonConstant.DEFAULT_SORT)
                            .map(this::buildEmoticonDto).collect(Collectors.toList());

                    return new EmoticonWrapper(buildEmoticonPackDto(emoticonPack), emoticons);
                }).collect(Collectors.toList());
    }

    public Page<EmoticonPackDto> getAllEmoticonPacks(Long channelId, Pageable pageable) {
        return emoticonPackRepository.findByChannelIdAndRecommendedFalseAndEnabledTrue(channelId, pageable)
                .map(this::buildEmoticonPackDto);
    }

    public Page<EmoticonPackDto> getFavoriteEmoticonPacks(Long channelId, String deviceId, Pageable pageable) {
        return emoticonPackFavoriteRepository.findByEmoticonPackChannelIdAndDeviceId(channelId, deviceId, pageable)
                .map(source -> buildEmoticonPackDto(source.getEmoticonPack()));
    }

    public Page<EmoticonDto> getFavoriteEmoticons(Long channelId, String deviceId, Pageable pageable) {
        return emoticonFavoriteRepository.findByEmoticonChannelIdAndDeviceId(channelId, deviceId, pageable)
                .map(source -> buildEmoticonDto(source.getEmoticon()));
    }

    public Page<EmoticonDto> getPopularityEmoticons(Long channelId, Pageable pageable) {
        return emoticonPopularityRepository.findByEmoticonChannelIdAndEmoticonEnabledTrue(channelId, pageable)
                .map(source -> buildEmoticonDto(source.getEmoticon()));

    }

    public Page<EmoticonDto> getOneEmoticonPack(Long channelId, Long emoticonPackId, Pageable pageable) {
        return emoticonRepository.findByChannelIdAndEmoticonPackIdAndEnabledTrue(channelId, emoticonPackId, pageable)
                .map(this::buildEmoticonDto);
    }

    public void submitFeedback(Feedback feedback) {
        feedbackRepository.save(feedback);
    }

    public EmoticonPack findEmoticonPackById(Long id) {
        return emoticonPackRepository.findOne(id);
    }

    public Emoticon findEmoticonById(Long id) {
        return emoticonRepository.findOne(id);
    }

    public void addRecorder(Recorder recorder) {
        if (recorder.getType() == Type.EMOTICON) {
            emoticonPopularityRepository.updatePopularity(recorder.getSourceId(), recorder.getAction().getPopularity());
        }

        recorderRepository.save(recorder);
    }

    public EmoticonPackFavorite findEmoticonPackFavorite(Long emoticonPackId, String deviceId) {
        return emoticonPackFavoriteRepository.findByEmoticonPackIdAndDeviceId(emoticonPackId, deviceId);
    }

    public void addEmoticonPackFavorite(EmoticonPackFavorite emoticonPackFavorite) {
        emoticonPackFavoriteRepository.save(emoticonPackFavorite);
    }

    public void deleteEmoticonPackFavorite(EmoticonPackFavorite emoticonPackFavorite) {
        emoticonPackFavoriteRepository.delete(emoticonPackFavorite);
    }

    public EmoticonFavorite findEmoticonFavorite(Long emoticonId, String deviceId) {
        return emoticonFavoriteRepository.findByEmoticonIdAndDeviceId(emoticonId, deviceId);
    }

    public void addEmoticonFavorite(EmoticonFavorite emoticonFavorite) {
        emoticonFavoriteRepository.save(emoticonFavorite);
    }

    public void deleteEmoticonFavorite(EmoticonFavorite emoticonFavorite) {
        emoticonFavoriteRepository.delete(emoticonFavorite);
    }

    public AdImageDto getEnabledAd(Long channelId, DeviceType type, DeviceSize size) {
        AdImage adImage = adImageRepository.findFirstByAdChannelIdAndAdEnabledTrueAndDeviceTypeAndDeviceSize(channelId, type, size);
        // Android first find size param else find LS size
        if (adImage == null && type == DeviceType.ANDROID && size != DeviceSize.LS) {
            adImage = adImageRepository.findFirstByAdChannelIdAndAdEnabledTrueAndDeviceTypeAndDeviceSize(channelId, type, DeviceSize.LS);
        }

        return buildAdImageDto(adImage);
    }

    public Configuration getConfiguration(Long channelId) {
        return configurationRepository.findByChannelId(channelId).orElse(new Configuration());
    }

    public AndroidUpdaterDto getAndroidUpdater(Long channelId) {
        return buildAndroidUpdaterDto(androidUpdaterRepository.findFirstByChannelIdOrderByVersionDesc(channelId));
    }

}
