/**
 * Project: xl-risk-control
 * <p/>
 * File Created at 2015年3月20日
 * lizhaog
 * <p/>
 * Copyright 2014 XunLei.com Corporation Limited.
 * All rights reserved.
 * <p/>
 * This software is the confidential and proprietary information of
 * XunLei Company. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with XunLei.com.
 */
package com.xunlei.channel.riskcontrol.task;

import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Strings;
import com.xunlei.channel.db.dao.RiskControlTaskDAO;
import com.xunlei.channel.db.pojo.RiskControlTask;
import com.xunlei.channel.paycommon.util.string.StringUtils;
import com.xunlei.channel.riskcontrol.constants.Constants;
import com.xunlei.channel.riskcontrol.eval.RiskControlEvaluator;
import com.xunlei.channel.riskevaluator.context.EvalContext;

/**
 * 抽象的风控任务
 * 这里有个问题：由于我们没有对产生的任务进行持久化存储，如果任务运行到一半重启了程序，如何去进行重查。
 * 这里我们采取由业务方来决定是否重查方式来解决。也就是业务方重新去启动任务来风控。
 *
 * @author lizhaog
 */
public abstract class AbstractRiskTask<T> implements RiskTask<T> {

    private static final Logger logger = LoggerFactory.getLogger(AbstractRiskTask.class);

    @Autowired
    protected RiskControlTaskDAO riskControlTaskDAO;

    @Autowired
    protected RiskControlEvaluator<T> riskControlEvaluator;

    public AbstractRiskTask() {

    }

    /**
     * 该构造方法提供给风控重查任务new匿名内部类时使用
     *
     * @param riskControlTaskDAO
     * @param riskControlEvaluator
     */
    public AbstractRiskTask(RiskControlTaskDAO riskControlTaskDAO,
                            RiskControlEvaluator<T> riskControlEvaluator) {
        this.riskControlTaskDAO = riskControlTaskDAO;
        this.riskControlEvaluator = riskControlEvaluator;
    }

    /**
     * 该方法用于执行风控的全部过程，包括：
     * 1.根据taskName获取风控任务信息，包括元数据载入来源，风控规则等
     * 2.根据元数据来源，导入元数据
     * 3.循环处理元数据，进行eval
     * 4.eval完成后，如果有标志位需要更新，则执行更新，如根据ID去数据库拿数据的任务，在执行完成后，需要更新该ID标志，以便元数据可以顺序不遗漏拿
     */
    @Override
    public void execute(String taskName) throws Exception {
        // 1.根据taskName获取task的信息并进行基本参数校验
        RiskControlTask riskControlTask = validateRiskTaskInfo(taskName);
        // 2. loadData 将字符串用分隔符拆分，然后再用等号拆分key和value放到map里
        Map<String, Object> paramMap = StringUtils.stringToMap(riskControlTask.getTaskParams(),
                Constants.RISK_CONTROL_TASK_PARAMS_SEPARATOR);
        //获取PayOrder对象集合
        List<T> datas = loadData(riskControlTask, paramMap);
        // 3. 进行风控判断
        if (datas != null && datas.size() > 0) {
            try {
                // 3.1 对数据进行鉴别，这里还是要改为多线程执行
                // TODO 目前还是简单处理吧，datas的大小让定时任务的力度来控制
                for (T data : datas) {
                    //data是PayOrder
                    //paramMap是source_pay_order_ok_by_id
                    //rulevalue是
                    EvalContext context = initialEvalContext(data, paramMap,
                            riskControlTask.getRuleValue());
                    //EvalContext{id:'15112786070397280844',rule:txt,data:没数据的map}
                    riskControlEvaluator.evalAndProcessEvalResult(context, taskName, data);
                }

                // 3.2 成功执行完毕，进行善后
                afterEval(riskControlTask);
            } catch (Exception e) {
                logger.error("", e);
                throw new Exception("taskName:" + taskName + "execute failed,with errmsg:"
                        + e.getMessage());
            }
        } else {
            logger.info("execute taskName:{} find 0 source data to risk", taskName);
        }

    }

    protected RiskControlTask validateRiskTaskInfo(String taskName) throws Exception {
        RiskControlTask riskControlTask = riskControlTaskDAO.getRiskControlTaskByName(taskName);
        if (riskControlTask == null) { // 失败
            logger.error(
                    "taskName:{} execute failed,cant find taskinfo config in db:risk_control_task",
                    taskName);
            throw new Exception("taskName:" + taskName
                    + "execute failed,cant find taskinfo config in db:risk_control_task");
        }
        String rule = riskControlTask.getRuleValue();
        if (Strings.isNullOrEmpty(rule)) {
            logger.error(
                    "taskName:{} execute failed,invalid taskinfo[rule is empty] config in db:risk_control_task",
                    taskName);
            throw new Exception(
                    "taskName:"
                            + taskName
                            + "execute failed,invalid taskinfo[rule is empty] config in db:risk_control_task");
        }
        return riskControlTask;
    }

    protected abstract List<T> loadData(RiskControlTask riskControlTask,
                                        Map<String, Object> paramMap) throws Exception;

    protected abstract EvalContext initialEvalContext(T data, Map<String, Object> paramMap,
                                                      String rule) throws Exception;

    // 默认donothing
    protected void afterEval(RiskControlTask riskControlTask) throws Exception {
    }

    ;

}
