/**
 * Project: xl-risk-control
 * 
 * File Created at 2015年3月19日
 * lizhaog
 * 
 * Copyright 2014 XunLei.com Corporation Limited.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * XunLei Company. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with XunLei.com.
 */
package com.xunlei.channel.riskcontrol.eval;

import com.xunlei.channel.riskcontrol.alarm.service.AlarmService;
import com.xunlei.channel.riskcontrol.constants.Constants;
import com.xunlei.channel.riskcontrol.constants.RiskResult;
import com.xunlei.channel.riskevaluator.constants.EvaluationCode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.base.Strings;
import com.xunlei.channel.db.dao.RiskControlResultDAO;
import com.xunlei.channel.db.pojo.RiskControlResult;
import com.xunlei.channel.riskcontrol.common.SerializeUtil;
import com.xunlei.channel.riskcontrol.constants.AlarmLevel;
import com.xunlei.channel.riskevaluator.context.EvalContext;
import com.xunlei.channel.riskevaluator.result.EvalResult;
import com.xunlei.channel.riskevaluator.service.EvaluatorService;

/**
 * 对数据和规则进行风控，同时处理风控结果;
 * 抽出一个工具类，以便重查和正常的风控任务都能使用
 * 
 * @author lizhaog
 */
@Service
public class RiskControlEvaluator<T> {

    private static final Logger logger = LoggerFactory.getLogger(RiskControlEvaluator.class);

    @Autowired
    protected RiskControlResultDAO riskControlResultDAO;

    @Autowired
    protected EvaluatorService evaluatorService;

    @Autowired
    protected AlarmService alarmService;

    public void evalAndProcessEvalResult(EvalContext context, String taskName, T data)
            throws Exception {
        EvalResult result = evaluatorService.eval(context);
        if (logger.isDebugEnabled()) {
            logger.debug("evalAndProcessEvalResult method:EvalResult{} ",result);
        }
        if(null!=result) {// 对结果进行对应处理
            if (!result.isEvalSucceess()) { // 程序发生异常
                processFailResult(result, context.getId(), taskName, data,
                        AlarmLevel.RECHECK.getLevel());
            } else if (result.getErrorCode().equals(EvaluationCode.ERROR_01.getCode())) {//若结果为掉单或两边数据不一致，则立即预警处理
                processAlarmNowResult(result, context.getId(), taskName, data);
            } else if (result.isRuleSuccess()) { // 程序未发生异常，规则判断成功
                processSuccessResult(result, context.getId(), taskName, data);
            } else { // 程序未发生异常，规则判断失败
                processFailResult(result, context.getId(), taskName, data,
                        AlarmLevel.RECHECK.getLevel());
            }
        }
    }

    /**
     * 处理立即预警
     * @param result
     * @param contextId
     * @param taskName
     * @param data
     * @throws Exception
     */
    private void processAlarmNowResult(EvalResult result, String contextId, String taskName, T data) throws Exception {
        logger.info("id:{} risk control  alarm now with result:{}", contextId, result);
        // 首先查询失败记录是否存在
        RiskControlResult riskControlResult = riskControlResultDAO.getRiskControlResult(contextId,
                taskName);
        String errorCode = result.getErrorCode();
        String errorMsg = result.getErrorMsg();
        if (Strings.isNullOrEmpty(errorMsg)) {
            errorMsg = EvaluationCode.ERROR_01.getMessage();
        }
        if (null == riskControlResult) { // 说明是第一次错误
            riskControlResult = new RiskControlResult();
            riskControlResult.setAlarmLevel(AlarmLevel.ALARM_NOW.getLevel());
            riskControlResult.setData(SerializeUtil.serialize(data));
            riskControlResult.setDataId(contextId);
            riskControlResult.setFailCount(1);
            riskControlResult.setErrorCode(errorCode);
            riskControlResult.setResultDesc(errorMsg);
            riskControlResult.setTaskName(taskName);
            riskControlResult.setResultValue(RiskResult.FAIL.getCode());
            riskControlResultDAO.saveRiskControlResult(riskControlResult);
        }else {
            int failCount = riskControlResult.getFailCount();
            riskControlResult.setFailCount(failCount + 1);
            riskControlResult.setAlarmLevel(AlarmLevel.ALARM_NOW.getLevel());
            riskControlResult.setResultValue(RiskResult.FAIL.getCode());
            riskControlResult.setResultDesc(riskControlResult.getResultDesc()+";"+errorMsg);
            riskControlResult.setErrorCode(errorCode);
            riskControlResultDAO.updateRiskControlResult(riskControlResult);
        }
    }

    /**
     * 处理失败任务
     * @param result
     * @param contextId
     * @param taskName
     * @param data
     * @param alarmLevel
     * @throws Exception
     */
    private void processFailResult(EvalResult result, String contextId, String taskName, T data,
                                   String alarmLevel) throws Exception {
        logger.info("id:{} risk control  failed with result:{}", contextId, result);
        // 首先查询失败记录是否存在
        RiskControlResult riskControlResult = riskControlResultDAO.getRiskControlResult(contextId,
                                                                                        taskName);
        String errorCode = result.getErrorCode();
        String errorMsg = result.getErrorMsg();
        if (Strings.isNullOrEmpty(errorMsg)) {
            errorMsg = EvaluationCode.ERROR_04.getMessage();
        }
        if (null == riskControlResult) { // 说明是第一次错误
            riskControlResult = new RiskControlResult();
            riskControlResult.setAlarmLevel(alarmLevel);
            riskControlResult.setData(SerializeUtil.serialize(data));
            riskControlResult.setDataId(contextId);
            riskControlResult.setFailCount(1);
            riskControlResult.setErrorCode(errorCode);
            riskControlResult.setResultDesc(errorMsg);
            riskControlResult.setTaskName(taskName);
            riskControlResult.setResultValue(RiskResult.FAIL.getCode());
            riskControlResultDAO.saveRiskControlResult(riskControlResult);
        } else { // 不是第一次，更新错误信息和失败次数
            int failCount = riskControlResult.getFailCount()+1;
            riskControlResult.setFailCount(failCount);
            riskControlResult.setAlarmLevel(failCount >= Constants.LIMIT_FAILCOUNT?AlarmLevel.ALARM_NOW.getLevel():alarmLevel);// 失败为五次或以上就立即预警
            riskControlResult.setResultValue(RiskResult.FAIL.getCode());

            riskControlResult.setResultDesc(Strings.isNullOrEmpty(riskControlResult.getResultDesc())?
                                                errorMsg:riskControlResult.getResultDesc()+";"+errorMsg);
            riskControlResult.setErrorCode(errorCode);
            riskControlResultDAO.updateRiskControlResult(riskControlResult);
        }
    }


    /**
     * 处理成功任务
     * @param result
     * @param contextId
     * @param taskName
     * @param data
     * @throws Exception
     */
    private void processSuccessResult(EvalResult result, String contextId, String taskName, T data)
            throws Exception {
        logger.info("id:{} risk control successed", contextId);
        String msg = result.getErrorMsg();
        if (Strings.isNullOrEmpty(msg)) {
            msg = EvaluationCode.SUCCESS_00.getMessage();
        }
        // 首先查询失败记录是否存在
        RiskControlResult riskControlResult = riskControlResultDAO.getRiskControlResult(contextId,
                taskName);

        //成功数据不保存，只更新数据为成功
        if (null != riskControlResult) {
            riskControlResult.setAlarmLevel(AlarmLevel.RECHECK_SUCCESS.getLevel());
            riskControlResult.setResultValue(RiskResult.SUCCESS.getCode());
            riskControlResult.setErrorCode(null);
            riskControlResult.setResultDesc(riskControlResult.getResultDesc()+";"+msg);
            riskControlResultDAO.updateRiskControlResult(riskControlResult);
        }
    }

}
