/**
 * Project: xl-risk-control
 * 
 * File Created at 2015年3月17日
 * lizhaog
 * 
 * Copyright 2014 XunLei.com Corporation Limited.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * XunLei Company. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with XunLei.com.
 */
package com.xunlei.channel.riskcontrol.common;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xunlei.channel.db.dao.ConfigInfoDAO;
import com.xunlei.channel.db.pojo.ConfigInfo;
import com.xunlei.channel.paycommon.configcache.ConfigCacheClientProxy;
import com.xunlei.channel.paycommon.configcache.vo.CacheKey;
import com.xunlei.channel.paycommon.configcache.vo.CacheValue;
import com.xunlei.channel.paycommon.util.string.StringUtils;

/**
 * 配置信息工具类
 * 
 * @author lizhaog
 */
@Component
public class ConfigInfoUtils {

    private static final Logger logger = LoggerFactory.getLogger(ConfigInfoUtils.class);

    @Autowired
    ConfigInfoDAO               configInfoDAO;

    @Autowired
    ConfigCacheClientProxy      configCacheClientProxy;

    /**
     * 根据groupId和propertyKey取在用的配置的值，先查缓存，缓存查不到就查数据库
     * 
     * @param groupId
     * @param propertyKey
     * @return 如果找到就返回对应值;如果找不到就返回null
     */
    public String getConfigValue(String groupId, String propertyKey) {
        if (StringUtils.isAnyNullOrEmpty(groupId, propertyKey)) {
            logger.info("groupId:{} or propertyKey:{} is null,return null", groupId, propertyKey);
            return null;
        }
        CacheKey cacheKey = new CacheKey(groupId, propertyKey);
        CacheValue cacheValue = configCacheClientProxy.get(cacheKey);
        String value = null;
        if (cacheValue == null) {
            value = getConfigInfoFromDB(groupId, propertyKey);
            logger.debug("get groupId:{},propertyKey:{} from db with resp value:{}", groupId,
                    propertyKey, value);
            if (value != null) { // get 到了，插入缓存
                configCacheClientProxy.put(cacheKey, new CacheValue(value));
            }
        } else {
            value = cacheValue.getPropertyValue();
            logger.debug("get groupId:{},propertyKey:{} from cache with resp value:{}", groupId,
                    propertyKey, value);
        }
        return value;
    }

    /**
     * 根据groupId和propertyKey取在用的配置的值，先查缓存，缓存查不到就查数据库
     * 
     * @param groupId
     * @param propertyKey
     * @return 如果找到就返回对应值;如果找不到就返回null
     */
    public boolean updateConfigValue(String groupId, String propertyKey, String value) {
        if (StringUtils.isAnyNullOrEmpty(groupId, propertyKey, value)) {
            logger.info("groupId:{} or propertyKey:{} is null,return null", groupId, propertyKey);
            return false;
        }
        CacheKey cacheKey = new CacheKey(groupId, propertyKey);
        CacheValue cacheValue = new CacheValue(value);
        boolean updateCache = configCacheClientProxy.put(cacheKey, cacheValue);
        boolean updateDb = updateConfigInfoToDB(groupId, propertyKey, value);
        logger.info("updateCache result:{},updateDb result:{} for groupId:{},propertyKey:{}",
                updateCache, updateDb, groupId, propertyKey);
        return updateCache && updateDb;
    }

    /**
     * 拿数据库中该条记录的propertyValue的值
     * 
     * @param groupId
     * @param propertyKey
     * @return 数据库查不到就返回null
     */
    private String getConfigInfoFromDB(String groupId, String propertyKey) {
        ConfigInfo configInfo = new ConfigInfo();
        configInfo.setGroupId(groupId);
        configInfo.setInUse("Y");
        configInfo.setPropertyKey(propertyKey);

        List<ConfigInfo> list = configInfoDAO.getConfigInfo(configInfo);
        if (list.size() > 0) {
            return list.get(0).getPropertyValue();
        }
        return null;
    }

    private boolean updateConfigInfoToDB(String groupId, String propertyKey, String value) {
        ConfigInfo configInfo = new ConfigInfo();
        configInfo.setGroupId(groupId);
        configInfo.setInUse("Y");
        configInfo.setPropertyKey(propertyKey);

        try {
            List<ConfigInfo> list = configInfoDAO.getConfigInfo(configInfo);
            if (list.size() > 0) {
                configInfo = list.get(0);
                configInfo.setPropertyValue(value);
                configInfoDAO.updateConfigInfo(configInfo);
                return true;
            } else {
                logger.info("cant find configinfo in db by groupId:{},propertyKey:{}", groupId,
                        propertyKey);
                return false;
            }
        } catch (Exception e) {
            logger.error("", e);
            return false;
        }
    }
}
