/**
 * Project: xl-risk-control
 * File Created at 15-9-18
 * qi
 * Copyright 2014 XunLei.com Corporation Limited.
 * All rights reserved.
 * This software is the confidential and proprietary information of
 * XunLei Company. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with XunLei.com.
 */
package com.xunlei.channel.riskcontrol.alarm.service.impl;

import com.xunlei.channel.alarm.api.AlarmApi;
import com.xunlei.channel.alarmcenter.dbservice.pojo.AlarmData;
import com.xunlei.channel.alarmcenter.dbservice.pojo.AlarmItem;
import com.xunlei.channel.alarmcenter.receive.pojo.AlarmReceiveResult;
import com.xunlei.channel.riskcontrol.alarm.processor.MessageProcessor;
import com.xunlei.channel.riskcontrol.alarm.service.AlarmService;
import com.xunlei.channel.riskcontrol.alarm.util.TemplateUtil;
import com.xunlei.channel.riskcontrol.alarm.vo.AlarmStatusVo;
import com.xunlei.channel.riskcontrol.util.MapperFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * 预警服务
 *
 * @author <a href="http://xiongyingqi.com">qi</a>
 * @version 2015-09-18 16:52
 */
@Component
public class AlarmServiceImpl implements AlarmService, InitializingBean {
    private static final Logger logger = LoggerFactory.getLogger(AlarmServiceImpl.class);
    @Value("#{alarm['alarm.alarmId']}")
    private String alarmId;
    @Value("#{alarm['alarm.alarmName']}")
    private String alarmName;
    @Value("#{alarm['alarm.description']}")
    private String alarmDescription;
    @Value("#{alarm['alarm.strategy']}")
    private String alarmStrategy;
    @Value("#{alarm['alarm.owner']}")
    private String alarmOwner;
    @Value("#{alarm['alarm.inUse']}")
    private boolean alarmInUse;
    @Value("#{alarm['alarm.groupId']}")
    private String alarmGroupId;
    @Value("#{alarm['api.baseUrl']}")
    private String alarmBaseUrl;
    @Value("#{alarm['template.messageTemplate']}")
    public String template;
    @Value("#{alarm['template.successMessageTemplate']}")
    public String successMessageTemplate;
    private AlarmApi alarmApi;
    @Autowired
    private MessageProcessor messageProcessor;
    @Autowired
    private TemplateUtil templateUtil;

    /**
     * 初始化预警配置
     */
    private void initAlarm() {
        try {
            alarmApi = new AlarmApi(alarmBaseUrl);
        } catch (Exception e) {
            logger.error("init alarmApi error!", e);
        }
        try {
            AlarmItem alarmItem = new AlarmItem();
            alarmItem.setAlarmId(alarmId);
            alarmItem.setDescription(alarmDescription);
            alarmItem.setInUse(alarmInUse);
            alarmItem.setName(alarmName);
            alarmItem.setOwner(alarmOwner);
            alarmItem.setStrategy(alarmStrategy);
            alarmApi.setAlarmItem(alarmItem);
        } catch (Exception e) {
            logger.error("init AlarmService error! ", e);
        }

//        /**
//         * 初始化预警线程，不断的轮询
//         */
//        thread = new Thread() {
//            @Override
//            public void run() {
//                while (true) {
//                    alarm();
//                    try {
//                        Thread.sleep(checkInterval);
//                    } catch (InterruptedException e) {
//                        logger.error("", e);
//                    }
//                }
//            }
//        };
//        thread.start();
    }

    public String buildDetail(Map<String, Integer> detailMap) {
        if (detailMap == null) {
            return null;
        }
        StringBuilder builder = new StringBuilder();
        builder.append("\r\n");
        for (Map.Entry<String, Integer> entry : detailMap.entrySet()) {
            builder.append(entry.getKey());
            builder.append(": ");
            builder.append(entry.getValue());
            builder.append("\r\n");
        }
        builder.delete(builder.length() - 2, builder.length());// 去掉最后的换行
        return builder.toString();
    }


    /**
     * 预警核心方法，该方法会被定时任务调用
     * 插入定时任务：
     * INSERT INTO task_config(task_name, project_name, project_no, task_no, task_class, task_method, initial_delay, config_value, config_type, status, remark)
     * VALUES ('风控系统预警', '风控系统', '00007', '00007004', 'com.xunlei.channel.riskcontrol.alarm.service.impl.AlarmServiceImpl', 'alarm', 10000, 300000, 'delay', 'Y', NULL );
     */
    public void alarm() {
        AlarmStatusVo alarmStatusVo = messageProcessor.isAlarm();
        if (alarmStatusVo == null || !alarmStatusVo.isNeedAlarm()) {
            if (logger.isInfoEnabled()) {
                logger.info("alarm... not need alarm!");
            }
            return;
        }
        if (alarmStatusVo.isSuccess()) {
            if (logger.isInfoEnabled()) {
                logger.info("alarm...  check success! send template: {}", successMessageTemplate);
            }
            sendAlarm(successMessageTemplate);
        } else {
            Map<String, Integer> detailMap = alarmStatusVo.getDetailMap();
            Integer failOrderNum = alarmStatusVo.getFailSize();
            Map<String, String> params = new HashMap<String, String>();
            params.put("failTimes", alarmStatusVo.getCheckFailTimes() + "");
            params.put("failOrderNum", failOrderNum + "");
            params.put("detail", buildDetail(detailMap));
            //核查失败[第{failTimes}次预警]！\r\n失败订单：{failOrderNum}个\r\n详情：{detail}
            try {
                String s = templateUtil.buildTemplate(template, params);
                if (logger.isInfoEnabled()) {
                    logger.info("alarm... template: {}, detailMap: {}, result: {}", template,
                            detailMap, s);
                }
                sendAlarm(s);
            } catch (Exception e) {
                logger.error("", e);
            }
        }
    }

    /**
     * 发送预警消息
     *
     * @param content
     */
    private void sendAlarm(String content) {
        AlarmData alarmData = new AlarmData();
        alarmData.setAlarmId(alarmId);
        alarmData.setContent(content);
        alarmData.setGroupId(alarmGroupId);
        try {
            AlarmReceiveResult receiveResult = alarmApi.addAlarmData(alarmData);
            boolean b = alarmApi.isSuccess(receiveResult);
            if (b) {
                if (logger.isDebugEnabled()) {
                    logger.debug("sendAlarm... alarm success! alarmData: {}",
                            MapperFactory.getObjectMapper().writeValueAsString(alarmData));
                }
            } else {
                logger.error("sendAlarm... alarm failed! alarmData: {}",
                        MapperFactory.getObjectMapper().writeValueAsString(alarmData));
            }
        } catch (Exception e) {
            logger.error("", e);
        }

    }

    /**
     * Invoked by a BeanFactory after it has set all bean properties supplied
     * (and satisfied BeanFactoryAware and ApplicationContextAware).
     * <p>This method allows the bean instance to perform initialization only
     * possible when all bean properties have been set and to throw an
     * exception in the event of misconfiguration.
     *
     * @throws Exception in the event of misconfiguration (such
     *                   as failure to set an essential property) or if initialization fails.
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        initAlarm();
    }

    public static void main(String[] args) {
        long second = 1000;
        long minute = 60 * second;
        long hour = 60 * minute;
        System.out.println(hour);
    }
}
