/**
 * Project: xl-risk-control
 * File Created at 15-9-23
 * qi
 * Copyright 2014 XunLei.com Corporation Limited.
 * All rights reserved.
 * This software is the confidential and proprietary information of
 * XunLei Company. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with XunLei.com.
 */
package com.xunlei.channel.riskcontrol.alarm.processor;

import com.xunlei.channel.db.dao.RiskControlResultDAO;
import com.xunlei.channel.db.pojo.RiskControlResult;
import com.xunlei.channel.riskcontrol.alarm.vo.AlarmStatusVo;
import com.xunlei.channel.riskcontrol.constants.AlarmLevel;
import com.xunlei.channel.riskcontrol.constants.Constants;
import com.xunlei.channel.riskcontrol.constants.RiskResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 消息处理器
 *
 * @author <a href="http://xiongyingqi.com">qi</a>
 * @version 2015-09-23 10:36
 */
@Component
public class MessageProcessor {
    private static final Logger logger = LoggerFactory.getLogger(MessageProcessor.class);
    @Autowired
    private RiskControlResultDAO riskControlResultDAO;
    // 标记上一次是否核查成功（第几次核查）
    private int                  checkFailTimes;
    private boolean              lastCheckFailed;

    /**
     * 是否需要预警
     *
     * @return
     */
    public AlarmStatusVo isAlarm() {
        // 查询需要预警的结果
        List<RiskControlResult> results = riskControlResultDAO
                .getRiskControlResultByValueAndAlarmLevel(
                        RiskResult.FAIL.getCode(), new String[] { AlarmLevel.ALARM_NOW.getLevel() },
                        Constants.SOURCE_DATA_LIMIT_500);
        return checkFailTimes(results);
    }

    /**
     * 是否核查成功
     *
     * @return boolean
     */
    public boolean isSuccess(Collection<RiskControlResult> results) {
        return results.isEmpty();
    }

    /**
     * 处理具体的详细错误
     *
     * @return Map<String, Integer>
     */
    protected Map<String, Integer> processDetail(Collection<RiskControlResult> results) {
        Map<String, Integer> map = new HashMap<String, Integer>();
        for (RiskControlResult allValue : results) {
            String resultDesc = allValue.getResultDesc();
            if (resultDesc == null) {
                continue;
            }
            Integer integer = map.get(resultDesc);
            if (integer == null || integer == 0) {
                map.put(resultDesc, 1);
            } else {
                map.put(resultDesc, integer + 1);
            }
        }
        if (logger.isInfoEnabled()) {
            logger.info("processDetail... all values: {}, result: {}", results, map);
        }
        return map;
    }

    /**
     * 检查是否需要预警，预警规则：如果上一次核查失败，那么这一次一定预警
     *
     * @return
     */
    public AlarmStatusVo checkFailTimes(Collection<RiskControlResult> results) {
        boolean success = isSuccess(results);
        if (lastCheckFailed) { // 如果上一次核查失败
            if (success) { // 代表这次核查成功
                checkFailTimes = 0;
                lastCheckFailed = false;
            } else {
                checkFailTimes++;
                lastCheckFailed = true;
            }
            if (logger.isDebugEnabled()) {
                logger.debug("isNeedAlarm... checkFailTimes: {}, isSuccess: {}, return: true",
                        checkFailTimes,
                        isSuccess(results));
            }
            return new AlarmStatusVo(true, success, checkFailTimes, results.size(),
                    processDetail(results));
        } else {// 如果上一次核查成功
            if (success) {// 这次也核查成功，那么不需要预警了
                checkFailTimes = 0;
                lastCheckFailed = false;
                if (logger.isDebugEnabled()) {
                    logger.debug("isNeedAlarm... lastCheckFailed: {}, isSuccess: {}, return: false",
                            lastCheckFailed,
                            isSuccess(results));
                }
                return new AlarmStatusVo(false, success, checkFailTimes, results.size(),
                        processDetail(results));
            } else { // 新的核查失败记录
                checkFailTimes++;
                lastCheckFailed = true;
                logger.info("isNeedAlarm... lastCheckFailed: {}, isSuccess: {}, return: true",
                        lastCheckFailed,
                        isSuccess(results));
                return new AlarmStatusVo(true, success, checkFailTimes, results.size(),
                        processDetail(results));
            }
        }
    }

    public int getCheckFailTimes() {
        return checkFailTimes;
    }

    public void setCheckFailTimes(int checkFailTimes) {
        this.checkFailTimes = checkFailTimes;
    }
}
