package com.xunlei.netty.httpserver.component;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Vector;

import org.jboss.netty.channel.ChannelFuture;
import org.jboss.netty.channel.ChannelFutureListener;
import org.jboss.netty.channel.ChannelHandlerContext;
import org.jboss.netty.handler.codec.http.HttpHeaders;
import org.jboss.netty.handler.codec.http.HttpVersion;
import org.slf4j.Logger;

import com.xunlei.netty.httpserver.cmd.CmdMappers.CmdMeta;
import com.xunlei.netty.httpserver.exception.ClosedChannelError;
import com.xunlei.netty.httpserver.exception.ProcessTimeoutError;
import com.xunlei.netty.httpserver.util.Log;

/**
 * <pre>
 * 开始解码		attach.decode (req.createTime)		
 * 									} 解码用时
 * 开始业务处理	attach.process (resp.createTime)
 * 									} 业务处理用时
 * 开始编码		attach.encode
 * 									} 编码用时
 * 发送完成		attach.complete
 * 
 * 2011-03-19 整理优化attach
 * 
 * 在具体的业务disaptcher中
 * 
 * channelOpen时,新建attach 这里直接注册一样生命周期的context  可选：[channelBound mark lastIoTime］[channelConn时,说明其准备 decode mark lastIoTime]
 * messageReceived时,说明其decode完毕,mark lastIoTime (这里能通过httpReq来找到其decode的开始时间,其resp的createTime就是跟当前lastIoTime一致),并注册最新的httpReq,httpResp(可以注册最新的messageEvent)
 * 
 * disaptch时,注册其业务线程,说明此线程可能会有长时间业务操作
 * 业务处理完成时,注销其业务线程
 * 真正wirteResponse时,是其开始encode的信号,因此mark lastIoTime(写)
 * 
 * writeComplete时,说明其已经encode完成,mark lastIoTime(写)
 *  
 * channelClose时,直接注销此attach,这是其生命周期终点  
 *                 
 * 
 * 在外部打描器中,发现有业务线程,则通过判断cmdMeta的timeout来判断是否中断
 * 没有业务线程,则是判断其ioTime  
 * 
 * @author ZengDong
 * @since 2010-5-22 下午09:29:54
 */
public class XLContextAttachment implements ChannelFutureListener {

	private static final Logger log = Log.getLogger();

	/**
	 * 保存了最近一次当前channel的request对象
	 */
	private XLHttpRequest request;
	/**
	 * 保存了最近一次当前channel的request对象
	 */
	private XLHttpResponse response;
	/**
	 * attach跟ctx的生命周期是一致的,在初始化时绑定
	 */
	private ChannelHandlerContext channelHandlerContext;
	/**
	 * 保存最近一次在dispatch中使用的cmdMeta
	 */
	private CmdMeta cmdMeta;
	/**
	 * 设置其业务线程
	 */
	private volatile Thread processThread;
	/**
	 * <pre>
	 * 保存最近一次请求
	 * 在处理过程中,发生错误时,都收集到这里
	 */
	private List<Throwable> throwables;
	private boolean closeAfterOperationComplete = true;
	private int respOperationCompleteCount = 0; // 2012-06-01 当前attach正常处理了几个回包，用于打开keep-alive时，统计回包数
	private static final String HTTP_HEADER_KEEP_ALIVE = "Keep-Alive";

	public void registerThrowable(Throwable ex) {
		if (throwables == null) {
			throwables = new ArrayList<Throwable>(1);
		}
		throwables.add(ex);
	}

	/**
	 * 业务线程自身check
	 */
	public void checkChannelOrThread() { // 业务线程自身check
		boolean shutdown = false;
		try {
			shutdown = !channelHandlerContext.getChannel().isOpen();
			if (shutdown) {
				throw ClosedChannelError.INSTANCE;
				// } else if (shutdown = (processThread != null && processThread.isInterrupted()) || Thread.currentThread().isInterrupted()) {
			}
			if (processThread != null) {
				shutdown = processThread.isInterrupted(); // 2012-04-16 zengdong 这里不能用 Thread.interrupt因为需要判断的是 正在处理的线程
				if (shutdown) {
					throw ProcessTimeoutError.INSTANCE;
				}
			}
		} finally {
			if (shutdown) {
				// 连接已经关了,或者已经被中断了
				//closeCloseable();
			}
		}
	}

	/**
	 * 在通道关闭时通知此attach进行中断及关闭
	 * 
	 * @return true表示是在 收包->发包中间被中断的(一般是由peer端)，false表示其他情况
	 */
	// 这里要同步的原因是怕在判断间隙此thread被业务使用,造成interrupt一个正常的线程了
	public synchronized void interrupt(StringBuilder info) {// 外部sweeper来触发中断
		if ( processThread != null) {
			processThread.interrupt();
			if (info != null) {
				info.append("[processThread]").append(processThread.getName());
			}
		}
	}

	public String getName() {
		StringBuilder sb = new StringBuilder();
		if (respOperationCompleteCount > 0) {
			sb.append("[").append(respOperationCompleteCount).append("]");
		}
		sb.append(cmdMeta);
		sb.append(request == null ? "" : "-" + Integer.toHexString(request.hashCode()));
		sb.append(channelHandlerContext.getChannel().getRemoteAddress());
		sb.append("/");
		// getTimeSpanInfo(sb, System.currentTimeMillis(), false);
		return sb.toString();
	}

	@Override
	public String toString() {
		return getName();
	}

	public XLContextAttachment(ChannelHandlerContext channelHandlerContext) {
		this.channelHandlerContext = channelHandlerContext;
	}

	/**
	 * @param request Http请求
	 */
	public void setRequest(XLHttpRequest request) {
		this.request = request;
	}

	/**
	 * Http请求
	 * @return
	 */
	public XLHttpRequest getRequest() {// TODO:此方法很多地方都要调用，看是否直接在这里面 checkChannelCLosed
		return request;
	}

	/**
	 * @param response Http响应
	 */
	public void setResponse(XLHttpResponse response) {
		this.response = response;
	}

	/**
	 * Http响应
	 * @return
	 */
	public XLHttpResponse getResponse() {
		return response;
	}

	public CmdMeta getCmdMeta() {
		return cmdMeta;
	}

	public void setCmdMeta(CmdMeta cmdMeta) {
		this.cmdMeta = cmdMeta;
	}

	public ChannelHandlerContext getChannelHandlerContext() {
		return channelHandlerContext;
	}

	public List<Throwable> getThrowables() {
		return throwables;
	}

	public Thread getProcessThread() {
		return processThread;
	}

	/**
	 * 注册当前业务处理的线程
	 */
	public synchronized void registerProcessThread() {
		//Thread.interrupted();// 重置当前线程的中断标志位
		this.processThread = Thread.currentThread();
	}

	/**
	 * 解除当前业务处理的线程
	 */
	public synchronized void unregisterProcessThread() {
		this.processThread = null;
	}

	public void operationComplete(ChannelFuture future) throws Exception {
		respOperationCompleteCount++;
		if (closeAfterOperationComplete) {
			future.getChannel().close();
		}
		if (request != null) {
			request.clean();
		}
	}

	public boolean setKeepAliveHeader() {
		// int timeout = serverConfig.getKeepAliveTimeout();
		// int responseSettedTimeout = response.getKeepAliveTimeout();
		int timeout = response.getKeepAliveTimeout();
		boolean close = true;
		try {
			if (timeout == -1) {// B.由客户端来决定
				close = HttpHeaders.Values.CLOSE.equalsIgnoreCase(request.getHeader(HttpHeaders.Names.CONNECTION))
						|| request.getProtocolVersion().equals(HttpVersion.HTTP_1_0)
						&& !HttpHeaders.Values.KEEP_ALIVE.equalsIgnoreCase(request.getHeader(HttpHeaders.Names.CONNECTION));
			} else {// C.使用全局配置
				close = timeout <= 0;
			}
		} finally {
			// if (!close)
			// There's no need to add 'Content-Length' header if this is the last response.
			// 为了兼容c的请求依赖content-length,不管是否close都带上length
			response.setHeader(HttpHeaders.Names.CONTENT_LENGTH, String.valueOf(response.getContentLength()));

			if (close) {
				response.setHeader(HttpHeaders.Names.CONNECTION, HttpHeaders.Values.CLOSE);
			} else {
				if (timeout != -1) {
					response.setHeader(HttpHeaders.Names.CONNECTION, HttpHeaders.Values.KEEP_ALIVE);
					response.setHeader(HTTP_HEADER_KEEP_ALIVE, "timeout=" + timeout);
				}
			}
		}
		closeAfterOperationComplete = close;
		return close;
	}
	
}